# Copyright 2022-2024 ETSI OSG/SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import grpc, logging
from typing import Iterator

import grpc._channel
from common.method_wrappers.Decorator import MetricsPool, safe_and_metered_rpc_method
from common.proto.context_pb2 import QoDConstraintsRequest, Constraint, ConstraintActionEnum, Constraint_QoSProfile, Constraint_Schedule, Empty, QoSProfileId, QoSProfile
from common.proto.qos_profile_pb2_grpc import QoSProfileServiceServicer
from context.client.ContextClient import ContextClient


LOGGER = logging.getLogger(__name__)

METRICS_POOL = MetricsPool('QoSProfile', 'RPC')

class QoSProfileServiceServicerImpl(QoSProfileServiceServicer):
    def __init__(self ) -> None:
        LOGGER.debug('Servicer Created')

    @safe_and_metered_rpc_method(METRICS_POOL, LOGGER)
    def CreateQoSProfile(self, request: QoSProfile, context: grpc.ServicerContext) -> QoSProfile:
        context_client = ContextClient()
        try:
            qos_profile_get = context_client.GetQoSProfile(request.qos_profile_id)
            context.set_details(f'QoSProfile {request.qos_profile_id.qos_profile_id.uuid} already exists')
            context.set_code(grpc.StatusCode.ALREADY_EXISTS)
            return QoSProfile()
        except grpc._channel._InactiveRpcError as exc:
            if exc.code() != grpc.StatusCode.NOT_FOUND:
                raise exc
        qos_profile = context_client.CreateQoSProfile(request)
        return qos_profile

    @safe_and_metered_rpc_method(METRICS_POOL, LOGGER)
    def UpdateQoSProfile(self, request: QoSProfile, context: grpc.ServicerContext) -> QoSProfile:
        context_client = ContextClient()
        try:
            _ = context_client.GetQoSProfile(request.qos_profile_id)
        except grpc._channel._InactiveRpcError as exc:
            if exc.code() == grpc.StatusCode.NOT_FOUND:
                context.set_details(f'QoSProfile {request.qos_profile_id.qos_profile_id.uuid} not found')
                context.set_code(grpc.StatusCode.NOT_FOUND)
                return QoSProfile()
        qos_profile = context_client.UpdateQoSProfile(request)
        return qos_profile

    @safe_and_metered_rpc_method(METRICS_POOL, LOGGER)
    def DeleteQoSProfile(self, request: QoSProfileId, context: grpc.ServicerContext) -> Empty:
        context_client = ContextClient()
        try:
            _ = context_client.GetQoSProfile(request)
        except grpc._channel._InactiveRpcError as exc:
            if exc.code() == grpc.StatusCode.NOT_FOUND:
                context.set_details(f'QoSProfile {request.qos_profile_id.uuid} not found')
                context.set_code(grpc.StatusCode.NOT_FOUND)
                return QoSProfile()
        empty = context_client.DeleteQoSProfile(request)
        return empty

    @safe_and_metered_rpc_method(METRICS_POOL, LOGGER)
    def GetQoSProfile(self, request: QoSProfileId, context: grpc.ServicerContext) -> QoSProfile:
        context_client = ContextClient()
        try:
            qos_profile = context_client.GetQoSProfile(request)
        except grpc._channel._InactiveRpcError as exc:
            if exc.code() == grpc.StatusCode.NOT_FOUND:
                context.set_details(f'QoSProfile {request.qos_profile_id.uuid} not found')
                context.set_code(grpc.StatusCode.NOT_FOUND)
                return QoSProfile()
        return qos_profile

    @safe_and_metered_rpc_method(METRICS_POOL, LOGGER)
    def GetQoSProfiles(self, request: Empty, context: grpc.ServicerContext) -> Iterator[QoSProfile]:
        context_client = ContextClient()
        yield from context_client.GetQoSProfiles(request)


    @safe_and_metered_rpc_method(METRICS_POOL, LOGGER)
    def GetConstraintListFromQoSProfile(self, request: QoDConstraintsRequest, context: grpc.ServicerContext) -> Iterator[Constraint]:
        context_client = ContextClient()
        try:
            qos_profile = context_client.GetQoSProfile(request.qos_profile_id)
        except grpc._channel._InactiveRpcError as exc:
            if exc.code() == grpc.StatusCode.NOT_FOUND:
                context.set_details(f'QoSProfile {request.qos_profile_id.qos_profile_id.uuid} not found')
                context.set_code(grpc.StatusCode.NOT_FOUND)
                yield Constraint()

        qos_profile_constraint = Constraint_QoSProfile()
        qos_profile_constraint.target_min_upstream_rate.CopyFrom(qos_profile.targetMinUpstreamRate)
        qos_profile_constraint.max_upstream_rate.CopyFrom(qos_profile.maxUpstreamRate)
        qos_profile_constraint.max_upstream_burst_rate.CopyFrom(qos_profile.maxUpstreamBurstRate)
        qos_profile_constraint.target_min_downstream_rate.CopyFrom(qos_profile.targetMinDownstreamRate)
        qos_profile_constraint.max_downstream_rate.CopyFrom(qos_profile.maxDownstreamRate)
        qos_profile_constraint.max_downstream_burst_rate.CopyFrom(qos_profile.maxDownstreamBurstRate)
        qos_profile_constraint.min_duration.CopyFrom(qos_profile.minDuration)
        qos_profile_constraint.max_duration.CopyFrom(qos_profile.maxDuration)
        qos_profile_constraint.priority = qos_profile.priority
        qos_profile_constraint.packet_delay_budget.CopyFrom(qos_profile.packetDelayBudget)
        qos_profile_constraint.jitter.CopyFrom(qos_profile.jitter)
        qos_profile_constraint.packet_error_loss_rate =qos_profile.packetErrorLossRate
        constraint_qos = Constraint()
        constraint_qos.action = ConstraintActionEnum.CONSTRAINTACTION_SET
        constraint_qos.qos_profile.CopyFrom(qos_profile_constraint)
        yield constraint_qos
        constraint_schedule = Constraint()
        constraint_schedule.action = ConstraintActionEnum.CONSTRAINTACTION_SET
        constraint_schedule.schedule.CopyFrom(Constraint_Schedule(start_timestamp=request.start_timestamp, duration_days=request.duration/86400))
        yield constraint_schedule
