# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import logging
from flask import request
from flask.json import jsonify
from flask_restful import Resource
from common.Constants import DEFAULT_CONTEXT_UUID
from common.proto.context_pb2 import ServiceId, ServiceStatusEnum, SliceStatusEnum
from context.client.ContextClient import ContextClient
from service.client.ServiceClient import ServiceClient
from .tools.Authentication import HTTP_AUTH
from .tools.ContextMethods import get_service, get_slice
from .tools.HttpStatusCodes import HTTP_GATEWAYTIMEOUT, HTTP_NOCONTENT, HTTP_OK, HTTP_SERVERERROR

LOGGER = logging.getLogger(__name__)

class L2VPN_Service(Resource):
    @HTTP_AUTH.login_required
    def get(self, vpn_id : str):
        LOGGER.debug('VPN_Id: {:s}'.format(str(vpn_id)))
        LOGGER.debug('Request: {:s}'.format(str(request)))

        # TODO: HACK ECOC'22, to be corrected
        response = jsonify({})
        response.status_code = HTTP_OK
        return response

        try:
            context_client = ContextClient()

            target = get_service(context_client, vpn_id)
            if target is not None:
                if target.service_id.service_uuid.uuid != vpn_id: # pylint: disable=no-member
                    raise Exception('Service retrieval failed. Wrong Service Id was returned')
                service_ready_status = ServiceStatusEnum.SERVICESTATUS_ACTIVE
                service_status = target.service_status.service_status # pylint: disable=no-member
                response.status_code = HTTP_OK if service_status == service_ready_status else HTTP_GATEWAYTIMEOUT
                return response

            target = get_slice(context_client, vpn_id)
            if target is not None:
                if target.slice_id.slice_uuid.uuid != vpn_id: # pylint: disable=no-member
                    raise Exception('Slice retrieval failed. Wrong Slice Id was returned')
                slice_ready_status = SliceStatusEnum.SLICESTATUS_ACTIVE
                slice_status = target.slice_status.slice_status # pylint: disable=no-member
                response.status_code = HTTP_OK if slice_status == slice_ready_status else HTTP_GATEWAYTIMEOUT
                return response

            raise Exception('VPN({:s}) not found in database'.format(str(vpn_id)))
        except Exception as e: # pylint: disable=broad-except
            LOGGER.exception('Something went wrong Retrieving VPN({:s})'.format(str(request)))
            response = jsonify({'error': str(e)})
            response.status_code = HTTP_SERVERERROR
        return response

    @HTTP_AUTH.login_required
    def delete(self, vpn_id : str):
        LOGGER.debug('VPN_Id: {:s}'.format(str(vpn_id)))
        LOGGER.debug('Request: {:s}'.format(str(request)))

        # pylint: disable=no-member
        service_id_request = ServiceId()
        service_id_request.context_id.context_uuid.uuid = DEFAULT_CONTEXT_UUID
        service_id_request.service_uuid.uuid = vpn_id

        try:
            service_client = ServiceClient()
            service_client.DeleteService(service_id_request)
            response = jsonify({})
            response.status_code = HTTP_NOCONTENT
        except Exception as e: # pylint: disable=broad-except
            LOGGER.exception('Something went wrong Deleting Service {:s}'.format(str(request)))
            response = jsonify({'error': str(e)})
            response.status_code = HTTP_SERVERERROR
        return response
