# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


from flask import render_template, Blueprint, flash, session, redirect, url_for
from common.proto.context_pb2 import Empty, LinkId, LinkList, TopologyId
from common.tools.context_queries.EndPoint import get_endpoint_names
from common.tools.object_factory.Context import json_context_id
from common.tools.object_factory.Topology import json_topology_id
from context.client.ContextClient import ContextClient


link = Blueprint('link', __name__, url_prefix='/link')
context_client = ContextClient()

@link.get('/')
def home():
    if 'context_uuid' not in session or 'topology_uuid' not in session:
        flash("Please select a context!", "warning")
        return redirect(url_for("main.home"))

    context_uuid = session['context_uuid']
    topology_uuid = session['topology_uuid']

    context_client.connect()
    json_topo_id = json_topology_id(topology_uuid, context_id=json_context_id(context_uuid))
    grpc_topology = context_client.GetTopology(TopologyId(**json_topo_id))
    topo_link_uuids = {link_id.link_uuid.uuid for link_id in grpc_topology.link_ids}
    grpc_links: LinkList = context_client.ListLinks(Empty())

    endpoint_ids = []
    links = []
    for link_ in grpc_links.links:
        if link_.link_id.link_uuid.uuid not in topo_link_uuids: continue
        links.append(link_)
        endpoint_ids.extend(link_.link_endpoint_ids)

    device_names, endpoints_data = get_endpoint_names(context_client, endpoint_ids)
    context_client.close()

    return render_template('link/home.html', links=links, device_names=device_names, endpoints_data=endpoints_data)


@link.route('detail/<path:link_uuid>', methods=('GET', 'POST'))
def detail(link_uuid: str):
    request = LinkId()
    request.link_uuid.uuid = link_uuid  # pylint: disable=no-member
    context_client.connect()
    response = context_client.GetLink(request)
    device_names, endpoints_data = get_endpoint_names(context_client, response.link_endpoint_ids)
    context_client.close()
    return render_template('link/detail.html',link=response, device_names=device_names, endpoints_data=endpoints_data)
