import logging, os, signal, sys, threading
from prometheus_client import start_http_server
from common.database.Factory import get_database
from context.Config import GRPC_SERVICE_PORT, GRPC_MAX_WORKERS, GRPC_GRACE_PERIOD, LOG_LEVEL, RESTAPI_SERVICE_PORT, \
    RESTAPI_BASE_URL, METRICS_PORT
from context.service.ContextService import ContextService
from context.service.rest_server.Server import Server
from context.service.rest_server.resources.Context import Context

terminate = threading.Event()
logger = None

def signal_handler(signal, frame):
    global terminate, logger
    logger.warning('Terminate signal received')
    terminate.set()

def main():
    global terminate, logger

    grpc_service_port    = os.environ.get('CONTEXTSERVICE_SERVICE_PORT_GRPC', GRPC_SERVICE_PORT   )
    max_workers          = os.environ.get('MAX_WORKERS',                      GRPC_MAX_WORKERS    )
    grace_period         = os.environ.get('GRACE_PERIOD',                     GRPC_GRACE_PERIOD   )
    log_level            = os.environ.get('LOG_LEVEL',                        LOG_LEVEL           )
    restapi_service_port = os.environ.get('RESTAPI_SERVICE_PORT',             RESTAPI_SERVICE_PORT)
    restapi_base_url     = os.environ.get('RESTAPI_BASE_URL',                 RESTAPI_BASE_URL    )
    metrics_port         = os.environ.get('METRICS_PORT',                     METRICS_PORT        )

    logging.basicConfig(level=log_level)
    logger = logging.getLogger(__name__)

    signal.signal(signal.SIGINT,  signal_handler)
    signal.signal(signal.SIGTERM, signal_handler)

    logger.info('Starting...')

    # Start metrics server
    start_http_server(metrics_port)

    # Get database instance
    database = get_database()

    # Starting context service
    grpc_service = ContextService(database, port=grpc_service_port, max_workers=max_workers, grace_period=grace_period)
    grpc_service.start()

    rest_server = Server(port=restapi_service_port, base_url=restapi_base_url)
    rest_server.add_resource(
        Context, '/restconf/config/context', endpoint='api.context', resource_class_args=(database,))
    rest_server.start()

    # Wait for Ctrl+C or termination signal
    while not terminate.wait(timeout=0.1): pass

    logger.info('Terminating...')
    grpc_service.stop()
    rest_server.shutdown()
    rest_server.join()

    logger.info('Bye')
    return 0

if __name__ == '__main__':
    sys.exit(main())
