# Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import logging, time
from common.proto.context_pb2 import ContextId, Empty
from common.proto.monitoring_pb2 import KpiDescriptorList
from common.tests.LoadScenario import load_scenario_from_descriptor
from common.tools.grpc.Tools import grpc_message_to_json_string
from common.tools.object_factory.Context import json_context_id
from context.client.ContextClient import ContextClient
from device.client.DeviceClient import DeviceClient
from monitoring.client.MonitoringClient import MonitoringClient
from tests.Fixtures import context_client, device_client, monitoring_client # pylint: disable=unused-import

LOGGER = logging.getLogger(__name__)
LOGGER.setLevel(logging.DEBUG)

DESCRIPTOR_FILE = 'ofc22/descriptors_emulated.json'

def test_scenario_bootstrap(
    context_client : ContextClient, # pylint: disable=redefined-outer-name
    device_client : DeviceClient,   # pylint: disable=redefined-outer-name
) -> None:
    # ----- List entities - Ensure database is empty -------------------------------------------------------------------
    response = context_client.ListContexts(Empty())
    assert len(response.contexts) == 0

    response = context_client.ListDevices(Empty())
    assert len(response.devices) == 0

    response = context_client.ListLinks(Empty())
    assert len(response.links) == 0


    # ----- Load Scenario ----------------------------------------------------------------------------------------------
    descriptor_loader = load_scenario_from_descriptor(
        DESCRIPTOR_FILE, context_client, device_client, None, None)


    # ----- List entities - Ensure scenario is ready -------------------------------------------------------------------
    response = context_client.ListContexts(Empty())
    assert len(response.contexts) == descriptor_loader.num_contexts

    for context_uuid, num_topologies in descriptor_loader.num_topologies.items():
        response = context_client.ListTopologies(ContextId(**json_context_id(context_uuid)))
        assert len(response.topologies) == num_topologies

    response = context_client.ListDevices(Empty())
    assert len(response.devices) == descriptor_loader.num_devices

    response = context_client.ListLinks(Empty())
    assert len(response.links) == descriptor_loader.num_links

    for context_uuid, _ in descriptor_loader.num_services.items():
        response = context_client.ListServices(ContextId(**json_context_id(context_uuid)))
        assert len(response.services) == 0

def test_scenario_kpis_created(
    context_client : ContextClient,         # pylint: disable=redefined-outer-name
    monitoring_client: MonitoringClient,    # pylint: disable=redefined-outer-name
) -> None:
    """
    This test validates that KPIs related to the service/device/endpoint were created
    during the service creation process.
    """
    response = context_client.ListDevices(Empty())
    kpis_expected = set()
    for device in response.devices:
        device_uuid = device.device_id.device_uuid.uuid
        for endpoint in device.device_endpoints:
            endpoint_uuid = endpoint.endpoint_id.endpoint_uuid.uuid
            for kpi_sample_type in endpoint.kpi_sample_types:
                kpis_expected.add((device_uuid, endpoint_uuid, kpi_sample_type))
    num_kpis_expected = len(kpis_expected)
    LOGGER.info('Num KPIs expected: {:d}'.format(num_kpis_expected))

    num_kpis_created, num_retry = 0, 0
    while (num_kpis_created != num_kpis_expected) and (num_retry < 5):
        response: KpiDescriptorList = monitoring_client.GetKpiDescriptorList(Empty())
        num_kpis_created = len(response.kpi_descriptor_list)
        LOGGER.info('Num KPIs created: {:d}'.format(num_kpis_created))
        time.sleep(0.5)
        num_retry += 1
    assert num_kpis_created == num_kpis_expected
