#!/bin/bash
# Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# Run from ~/tfs-ctrl as:
# python scripts/update_license_headers.py

import logging, os, re, sys
from io import TextIOWrapper

logging.basicConfig(level=logging.INFO)
LOGGER = logging.getLogger(__name__)

ROOT_PATH = '.'
FILE_PATH_SKIPPED    = 'out-skipped.txt'
FILE_PATH_NO_HEADER  = 'out-no-header.txt'
FILE_PATH_UPDATED    = 'out-updated.txt'

#RE_OLD_COPYRIGHT  = re.compile(r'Copyright\ 2021\-2023\ H2020\ TeraFlow\ \(https\:\/\/www\.teraflow\-h2020\.eu\/\)')
RE_OLD_COPYRIGHT  = re.compile(r'Copyright\ 2022\-2023\ ETSI\ TeraFlowSDN\ \-\ TFS\ OSG\ \(https\:\/\/tfs\.etsi\.org\/\)')
STR_NEW_COPYRIGHT = 'Copyright 2022-2024 ETSI TeraFlowSDN - TFS OSG/SDG (https://tfs.etsi.org/)'

def skip_file(file_path : str) -> bool:
    if file_path.endswith('.pyc'): return True
    if file_path.endswith('_pb2_grpc.py'): return True
    if file_path.endswith('_pb2.py'): return True
    if file_path.endswith('.md'): return True
    if file_path.endswith('.png'): return True
    if file_path.endswith('.json'): return True
    if file_path.endswith('.zip'): return True
    if file_path.endswith('.zip'): return True
    if file_path.endswith('.jar'): return True
    if file_path.endswith('/tstat'): return True
    if file_path.endswith('/.gitignore'): return True
    if file_path.endswith('/.gitkeep'): return True
    if file_path.endswith('/coverage/.coverage'): return True
    if file_path.startswith('./netconf_openconfig/'): return True
    if file_path.startswith('./tmp/'): return True
    if '/manifests/cttc-ols/' in file_path: return True
    if '/hackfest/netconf-oc/openconfig/' in file_path: return True
    if '/.git/' in file_path: return True
    if '/.vscode/' in file_path: return True
    if '/.pytest_cache/' in file_path: return True
    if '/__pycache__/' in file_path: return True
    if '/.mvn/' in file_path: return True
    if '/hackfest/kafka/kafka_2.13-2.8.0' in file_path: return True
    if '/src/device/service/drivers/openconfig/templates/' in file_path: return True
    if '/ztp/target/generated-sources/grpc/' in file_path: return True
    if '/ztp/target/kubernetes/' in file_path: return True
    if '/policy/target/generated-sources/grpc/' in file_path: return True
    if '/policy/target/kubernetes/' in file_path: return True
    if '/src/dlt/gateway/' in file_path: return True
    if FILE_PATH_SKIPPED in file_path: return True
    if FILE_PATH_NO_HEADER in file_path: return True
    if FILE_PATH_UPDATED in file_path: return True
    if file_path in {'./LICENSE', './.python-version', './.env'}: return True
    return False

def process_line(line_in : str) -> str:
    line_out = RE_OLD_COPYRIGHT.sub(STR_NEW_COPYRIGHT, line_in)
    if line_out != line_in: return line_out
    return line_in

def process_file(
    file_path : str, file_no_header : TextIOWrapper, file_skipped : TextIOWrapper, file_updated : TextIOWrapper
) -> None:
    if skip_file(file_path):
        file_skipped.write(file_path + '\n')
        return

    LOGGER.info('  File {:s}...'.format(str(file_path)))

    temp_file_path = file_path + '.temp'
    replaced = False
    try:
        with open(file_path, encoding='UTF-8') as source:
            with open(temp_file_path, 'w', encoding='UTF-8') as target:
                for line_in in source:
                    line_out = process_line(line_in)
                    target.write(line_out)
                    replaced = replaced or (line_out != line_in)
    except: # pylint: disable=bare-except
        replaced = False

    if not replaced:
        file_no_header.write(file_path + '\n')
    else:
        file_updated.write(file_path + '\n')

    os.rename(temp_file_path, file_path)

def main() -> int:
    with open(FILE_PATH_NO_HEADER, 'w', encoding='UTF-8') as file_no_header:
        with open(FILE_PATH_SKIPPED, 'w', encoding='UTF-8') as file_skipped:
            with open(FILE_PATH_UPDATED, 'w', encoding='UTF-8') as file_updated:
                for dirpath, _, filenames in os.walk(ROOT_PATH):
                    LOGGER.info('Folder {:s}...'.format(str(dirpath)))
                    for filename in filenames:
                        file_path = os.path.join(dirpath, filename)
                        process_file(file_path, file_no_header, file_skipped, file_updated)
    return 0

if __name__ == '__main__':
    sys.exit(main())
