# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import logging, signal, sys, time, threading, random
from prometheus_client import start_http_server
from celery import Celery

from common.Constants import DEFAULT_GRPC_MAX_WORKERS, DEFAULT_GRPC_GRACE_PERIOD
from common.Settings import get_log_level, get_metrics_port, get_setting
from opticalattackdetector.Config import (
    GRPC_SERVICE_PORT, MONITORING_INTERVAL)
from common.proto.context_pb2 import (Empty,
    Context,  ContextId,  ContextIdList,  ContextList,
    Service,  ServiceId,  ServiceIdList,  ServiceList, Timestamp
)
from common.proto.dbscanserving_pb2 import DetectionRequest, DetectionResponse, Sample
from common.proto.attack_mitigator_pb2 import AttackDescription, AttackResponse
from dbscanserving.client.DbscanServingClient import DbscanServingClient
from opticalattackmitigator.client.OpticalAttackMitigatorClient import OpticalAttackMitigatorClient
from opticalattackdetector.service.OpticalAttackDetectorService import OpticalAttackDetectorService
from opticalattackdetector.client.OpticalAttackDetectorClient import OpticalAttackDetectorClient
from monitoring.client.MonitoringClient import MonitoringClient
from common.proto.monitoring_pb2 import Kpi

terminate = threading.Event()
LOGGER = None

client: OpticalAttackDetectorClient = None


def signal_handler(signal, frame): # pylint: disable=redefined-outer-name
    LOGGER.warning('Terminate signal received')
    terminate.set()


def detect_attack_old(monitoring_interval):
    time.sleep(10)  # wait for the service to start
    LOGGER.info("Starting the attack detection loop")
    client = OpticalAttackDetectorClient(address='localhost', port=GRPC_SERVICE_PORT)
    client.connect()
    while True:  # infinite loop that runs until the terminate is set
        if terminate.is_set():  # if terminate is set
            LOGGER.warning("Stopping execution...")
            client.close()
            break  # break the while and stop execution
        client.DetectAttack(Empty())
        # sleep
        LOGGER.debug("Sleeping for {} seconds...".format(monitoring_interval))
        time.sleep(monitoring_interval)

def main():
    global LOGGER # pylint: disable=global-statement

    log_level = get_log_level()
    logging.basicConfig(level=log_level)
    LOGGER = logging.getLogger(__name__)

    service_port = get_setting('OPTICALATTACKDETECTORSERVICE_SERVICE_PORT_GRPC', default=GRPC_SERVICE_PORT)
    max_workers  = get_setting('MAX_WORKERS',                                    default=DEFAULT_GRPC_MAX_WORKERS )
    grace_period = get_setting('GRACE_PERIOD',                                   default=DEFAULT_GRPC_GRACE_PERIOD)

    signal.signal(signal.SIGINT,  signal_handler)
    signal.signal(signal.SIGTERM, signal_handler)

    LOGGER.info('Starting...')

    # Start metrics server
    metrics_port = get_metrics_port()
    start_http_server(metrics_port)

    dbscanserving_client: DbscanServingClient = DbscanServingClient()
    attack_mitigator_client: OpticalAttackMitigatorClient = OpticalAttackMitigatorClient()
    monitoring_client: MonitoringClient = MonitoringClient()

    # Starting CentralizedCybersecurity service
    grpc_service = OpticalAttackDetectorService(
        port=service_port, max_workers=max_workers, grace_period=grace_period)
    grpc_service.start()

    # p = multiprocessing.Process(target=detect_attack, args=(monitoring_interval, ))
    # p.start()
    # detect_attack(monitoring_interval)

    LOGGER.info('Connecting with REDIS...')
    REDIS_PASSWORD = get_setting('REDIS_PASSWORD')
    REDIS_HOST = get_setting('CACHINGSERVICE_SERVICE_HOST')
    REDIS_PORT = get_setting('CACHINGSERVICE_SERVICE_PORT_REDIS')
    BROKER_URL = f'redis://:{REDIS_PASSWORD}@{REDIS_HOST}:{REDIS_PORT}/0'
    app = Celery(
        'cybersecurity',
        broker=BROKER_URL,
        backend=BROKER_URL
    )
    LOGGER.info('Connected to REDIS...')

    @app.task(name='detect_attack')
    def detect_attack(context_id, service_id, kpi_id):
        LOGGER.info('Detecting attack for {}/{}'.format(context_id, service_id))
        alien_samples = random.randint(2, 10)
        # run attack detection for every service
        request: DetectionRequest = DetectionRequest()
        request.num_samples = 300 + alien_samples
        request.num_features = 20
        request.eps = 100.5
        request.min_samples = 5
        for _ in range(200):
            grpc_sample = Sample()
            for __ in range(20):
                grpc_sample.features.append(random.uniform(0., 10.))
            request.samples.append(grpc_sample)
        for _ in range(100):
            grpc_sample = Sample()
            for __ in range(20):
                grpc_sample.features.append(random.uniform(50., 60.))
            request.samples.append(grpc_sample)                  
        for _ in range(alien_samples):
            grpc_sample = Sample()
            for __ in range(20):
                grpc_sample.features.append(random.uniform(5000., 6000.))
            request.samples.append(grpc_sample)
        response: DetectionResponse = dbscanserving_client.Detect(request)

        # including KPI
        kpi = Kpi()
        kpi.kpi_id.kpi_id.uuid = kpi_id
        kpi.timestamp = Timestamp()
        kpi.kpi_value.int32Val = response.cluster_indices[-1]
        monitoring_client.IncludeKpi(kpi)

        if -1 in response.cluster_indices:  # attack detected
            attack = AttackDescription()
            # attack.cs_id.uuid = service.service_id.service_uuid.uuid
            response: AttackResponse = attack_mitigator_client.NotifyAttack(attack)
        return 0
    
    app.worker_main([
        'worker',
        '--loglevel={}'.format(log_level)
    ])

    # Wait for Ctrl+C or termination signal
    while not terminate.wait(timeout=0.1): pass

    LOGGER.info('Terminating...')
    grpc_service.stop()
    # p.kill()

    LOGGER.info('Bye')
    return 0

if __name__ == '__main__':
    sys.exit(main())
