# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import os, grpc, logging, random
from common.rpc_method_wrapper.Decorator import create_metrics, safe_and_metered_rpc_method
from context.client.ContextClient import ContextClient
from monitoring.client.MonitoringClient import MonitoringClient
from service.client.ServiceClient import ServiceClient
from common.proto.dbscanserving_pb2 import DetectionRequest, DetectionResponse, Sample
from dbscanserving.client.DbscanServingClient import DbscanServingClient
from opticalattackmitigator.client.OpticalAttackMitigatorClient import OpticalAttackMitigatorClient
from common.proto.optical_attack_mitigator_pb2 import AttackDescription, AttackResponse
from common.proto.context_pb2 import (Empty,
    Context,  ContextId,  ContextIdList,  ContextList,
    Service,  ServiceId,  ServiceIdList,  ServiceList
)
from common.proto.monitoring_pb2 import KpiList
from common.proto.optical_attack_detector_pb2_grpc import (
    OpticalAttackDetectorServiceServicer)


LOGGER = logging.getLogger(__name__)

SERVICE_NAME = 'OpticalAttackDetector'
METHOD_NAMES = ['NotifyServiceUpdate', 'DetectAttack', 'ReportSummarizedKpi', 'ReportKpi']
METRICS = create_metrics(SERVICE_NAME, METHOD_NAMES)

context_client: ContextClient = ContextClient()
monitoring_client: MonitoringClient = MonitoringClient()
dbscanserving_client: DbscanServingClient = DbscanServingClient()
service_client: ServiceClient = ServiceClient()
attack_mitigator_client: OpticalAttackMitigatorClient = OpticalAttackMitigatorClient()


class OpticalAttackDetectorServiceServicerImpl(OpticalAttackDetectorServiceServicer):

    def __init__(self):
        LOGGER.debug('Creating Servicer...')
        LOGGER.debug('Servicer Created')

    @safe_and_metered_rpc_method(METRICS, LOGGER)
    def DetectAttack(self, request : Empty, context : grpc.ServicerContext) -> Empty:
        
        # retrieve list with current contexts
        # import pdb; pdb.set_trace()
        context_ids: ContextIdList = context_client.ListContextIds(Empty())

        # for each context, retrieve list of current services
        services = []
        for context_id in context_ids.context_ids:

            context_services: ServiceIdList = context_client.ListServices(context_id)
            for service in context_services.services:
                services.append(service)

        for service in services:
            for endpoint in service.service_endpoint_ids:
                # get instant KPI for this endpoint
                LOGGER.warning(f'service: {service.service_id.service_uuid.uuid}\t endpoint: {endpoint.endpoint_uuid.uuid}\tdevice: {endpoint.device_id.device_uuid.uuid}')

                # run attack detection for every service
                request: DetectionRequest = DetectionRequest()

                request.num_samples = 310
                request.num_features = 100
                request.eps = 100.5
                request.min_samples = 50

                for _ in range(200):
                    grpc_sample = Sample()
                    for __ in range(100):
                        grpc_sample.features.append(random.uniform(0., 10.))
                    request.samples.append(grpc_sample)
                    
                for _ in range(100):
                    grpc_sample = Sample()
                    for __ in range(100):
                        grpc_sample.features.append(random.uniform(50., 60.))
                    request.samples.append(grpc_sample)
                    
                for _ in range(10):
                    grpc_sample = Sample()
                    for __ in range(100):
                        grpc_sample.features.append(random.uniform(5000., 6000.))
                    request.samples.append(grpc_sample)

                response: DetectionResponse = dbscanserving_client.Detect(request)

                if -1 in response.cluster_indices:  # attack detected
                    attack = AttackDescription()
                    attack.cs_id.uuid = service.service_id.service_uuid.uuid
                    response: AttackResponse = attack_mitigator_client.NotifyAttack(attack)

        # if attack is detected, run the attack mitigator
        return Empty()

    @safe_and_metered_rpc_method(METRICS, LOGGER)
    def ReportSummarizedKpi(self, request : KpiList, context : grpc.ServicerContext) -> Empty:
        return Empty()

    @safe_and_metered_rpc_method(METRICS, LOGGER)
    def ReportKpi(self, request : KpiList, context : grpc.ServicerContext) -> Empty:
        return Empty()
