# Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from __future__ import print_function
import logging
from common.proto.l3_centralizedattackdetector_pb2 import (
    Empty
)
from common.proto.l3_attackmitigator_pb2_grpc import (
    L3AttackmitigatorServicer,
)
from common.proto.context_pb2 import (
    Service, ServiceId, ServiceConfig, ServiceTypeEnum, ServiceStatusEnum, ServiceStatus, Context, ContextId, Uuid,
    Timestamp, ConfigRule, ConfigRule_Custom, ConfigActionEnum, Device, DeviceId, DeviceConfig,
    DeviceOperationalStatusEnum, DeviceDriverEnum, EndPoint, Link, LinkId, EndPoint, EndPointId, Topology, TopologyId
)
from common.proto.context_pb2_grpc import (
    ContextServiceStub
)
from common.proto.service_pb2_grpc import (
    ServiceServiceStub
)
from datetime import datetime
import grpc

LOGGER = logging.getLogger(__name__)
CONTEXT_CHANNEL = "192.168.165.78:1010"
SERVICE_CHANNEL = "192.168.165.78:3030"

class l3_attackmitigatorServiceServicerImpl(L3AttackmitigatorServicer):

    def GetNewService(self, service_id):
        service = Service()
        service_id_obj = self.GenerateServiceId(service_id)
        service.service_id.CopyFrom(service_id_obj)
        service.service_type = ServiceTypeEnum.SERVICETYPE_L3NM
        service_status = ServiceStatus()
        service_status.service_status = ServiceStatusEnum.SERVICESTATUS_ACTIVE
        service.service_status.CopyFrom(service_status)
        timestamp = Timestamp()
        timestamp.timestamp = datetime.timestamp(datetime.now())
        service.timestamp.CopyFrom(timestamp)
        return service
    
    def GetNewContext(self, service_id):
        context = Context()
        context_id = ContextId()
        uuid = Uuid()
        uuid.uuid = service_id
        context_id.context_uuid.CopyFrom(uuid)
        context.context_id.CopyFrom(context_id)
        return context

    def GetNewDevice(self, service_id):
        device = Device()
        device_id = DeviceId()
        uuid = Uuid()
        uuid.uuid = service_id
        device_id.device_uuid.CopyFrom(uuid)
        device.device_type="test"
        device.device_id.CopyFrom(device_id)
        device.device_operational_status = DeviceOperationalStatusEnum.DEVICEOPERATIONALSTATUS_ENABLED
        return device

    def GetNewLink(self, service_id):
        link = Link()
        link_id = LinkId()
        uuid = Uuid()
        uuid.uuid = service_id
        link_id.link_uuid.CopyFrom(uuid)
        link.link_id.CopyFrom(link_id)
        return link

    def GetNewTopology(self,context_id, device_id, link_id):
        topology = Topology()
        topology_id = TopologyId()
        topology_id.context_id.CopyFrom(context_id)
        uuid = Uuid()
        uuid.uuid = "test_crypto"
        topology_id.topology_uuid.CopyFrom(uuid)
        topology.topology_id.CopyFrom(topology_id)
        topology.device_ids.extend([device_id])
        topology.link_ids.extend([link_id])
        return topology

    def GetNewEndpoint(self, topology_id, device_id, uuid_name):
        endpoint = EndPoint()
        endpoint_id = EndPointId()
        endpoint_id.topology_id.CopyFrom(topology_id)
        endpoint_id.device_id.CopyFrom(device_id)
        uuid = Uuid()
        uuid.uuid = uuid_name
        endpoint_id.endpoint_uuid.CopyFrom(uuid)
        endpoint.endpoint_id.CopyFrom(endpoint_id)
        endpoint.endpoint_type = "test"
        return endpoint
        

    def __init__(self):
        LOGGER.debug("Creating Servicer...")
        self.last_value = -1
        self.last_tag = 0
       """ 
        context = self.GetNewContext("test_crypto")
        print(context, flush=True)
        print(self.SetContext(context))

        service = self.GetNewService("test_crypto")
        print("This is the new service", self.CreateService(service), flush = True)

        ip_o = "127.0.0.1"
        ip_d = "127.0.0.2"
        port_o = "123"
        port_d = "124"

        service_id = self.GenerateServiceId("test_crypto")

        config_rule = self.GetConfigRule(ip_o, ip_d, port_o, port_d)

        service = self.GetService(service_id)
        print("Service obtained from id", service, flush=True)
        
        config_rule = self.GetConfigRule(ip_o, ip_d, port_o, port_d)

        #service_config = service.service_config
        #service_config.append(config_rule)
        
        service_config = ServiceConfig()
        service_config.config_rules.extend([config_rule])
        service.service_config.CopyFrom(service_config)
        
        device = self.GetNewDevice("test_crypto")
        print("New device", device, flush=True)
        device_id = self.SetDevice(device)

        link = self.GetNewLink("test_crypto")
        print("New link", link, flush=True)
        link_id = self.SetLink(link)
        
        topology = self.GetNewTopology(context.context_id, device.device_id, link.link_id)
        print("New topology", topology, flush=True)
        topology_id = self.SetTopology(topology)

        endpoint = self.GetNewEndpoint(topology.topology_id, device.device_id, "test_crypto")
        print("New endpoint", endpoint, flush=True)
        link.link_endpoint_ids.extend([endpoint.endpoint_id])

        self.SetLink(link)

        print("Service with new rule", service, flush=True)
        self.UpdateService(service)

        service2 = self.GetService(service_id)
        print("Service obtained from id after updating", service2, flush=True)
        """

    def GenerateRuleValue(self, ip_o, ip_d, port_o, port_d):
        value = {
            'ipv4:source-address': ip_o,
            'ipv4:destination-address': ip_d,
            'transport:source-port': port_o,
            'transport:destination-port': port_d,
            'forwarding-action': 'DROP',
        }
        return value

    def GetConfigRule(self, ip_o, ip_d, port_o, port_d):
        config_rule = ConfigRule()
        config_rule_custom = ConfigRule_Custom()
        config_rule.action = ConfigActionEnum.CONFIGACTION_SET
        config_rule_custom.resource_key = 'test'
        config_rule_custom.resource_value = str(self.GenerateRuleValue(ip_o, ip_d, port_o, port_d))
        config_rule.custom.CopyFrom(config_rule_custom)
        return config_rule

    def GenerateServiceId(self, service_id):
        service_id_obj = ServiceId()
        context_id = ContextId()
        uuid = Uuid()
        uuid.uuid = service_id
        context_id.context_uuid.CopyFrom(uuid)
        service_id_obj.context_id.CopyFrom(context_id)
        service_id_obj.service_uuid.CopyFrom(uuid)
        return service_id_obj
   
    def SendOutput(self, request, context):
        # SEND CONFIDENCE TO MITIGATION SERVER
        print("Server received mitigation values...", request.confidence, flush=True)
        
        last_value = request.confidence
        last_tag = request.tag

        ip_o = request.ip_o
        ip_d = request.ip_d
        port_o = request.port_o
        port_d = request.port_d

        service_id = self.GenerateServiceId(request.service_id)

        config_rule = self.GetConfigRule(ip_o, ip_d, port_o, port_d)
        
        service = GetService(service_id)
        print(service)
        #service.config_rules.append(config_rule)
        #UpdateService(service)

        # RETURN OK TO THE CALLER
        return Empty(
            message=f"OK, received values: {last_tag} with confidence {last_value}."
        )
     
    def SetDevice(self, device):
        with grpc.insecure_channel(CONTEXT_CHANNEL) as channel:
            stub = ContextServiceStub(channel)
            return stub.SetDevice(device)

    def SetLink(self, link):
        with grpc.insecure_channel(CONTEXT_CHANNEL) as channel:
            stub = ContextServiceStub(channel)
            return stub.SetLink(link)

    def SetTopology(self, link):
        with grpc.insecure_channel(CONTEXT_CHANNEL) as channel:
            stub = ContextServiceStub(channel)
            return stub.SetTopology(link)


    def GetService(self, service_id):
        with grpc.insecure_channel(CONTEXT_CHANNEL) as channel:
            stub = ContextServiceStub(channel)
            return stub.GetService(service_id)

    def SetContext(self, context):
        with grpc.insecure_channel(CONTEXT_CHANNEL) as channel:
            stub = ContextServiceStub(channel)
            return stub.SetContext(context)

    def UpdateService(self, service):
        with grpc.insecure_channel(SERVICE_CHANNEL) as channel:
            stub = ServiceServiceStub(channel)
            stub.UpdateService(service)

    def CreateService(self, service):
        with grpc.insecure_channel(SERVICE_CHANNEL) as channel:
            stub = ServiceServiceStub(channel)
            stub.CreateService(service)

    def GetMitigation(self, request, context):
        # GET OR PERFORM MITIGATION STRATEGY
        logging.debug("")
        print("Returing mitigation strategy...")
        k = self.last_value * 2
        return Empty(
            message=f"Mitigation with double confidence = {k}"
        )
    
