import grpc, logging
from enum import Enum
from typing import Dict, List, Set, Tuple
from common.Checkers import chk_options, chk_string
from common.database.api.Database import Database
from common.database.api.context.Constants import DEFAULT_TOPOLOGY_ID
from common.database.api.context.topology.device.Endpoint import Endpoint
from common.database.api.context.service.ServiceState import ServiceState, servicestate_enum_values, to_servicestate_enum
from common.database.api.context.service.ServiceType import ServiceType, servicetype_enum_values, to_servicetype_enum
from common.exceptions.ServiceException import ServiceException
from service.proto.context_pb2 import Constraint, EndPointId
from service.proto.service_pb2 import Service, ServiceId

# For each method name, define acceptable service types. Empty set means accept all.
ACCEPTED_SERVICE_TYPES : Dict[str, Set[ServiceType]] = {
    'CreateService': set([ServiceType.L2NM, ServiceType.L3NM, ServiceType.TAPI_CONNECTIVITY_SERVICE]),
    'UpdateService': set([ServiceType.L2NM, ServiceType.L3NM, ServiceType.TAPI_CONNECTIVITY_SERVICE]),
}

# For each method name, define acceptable service states. Empty set means accept all.
ACCEPTED_SERVICE_STATES : Dict[str, Set[ServiceState]] = {
    'CreateService': set([ServiceState.PLANNED]),
    'UpdateService': set([ServiceState.PLANNED, ServiceState.ACTIVE, ServiceState.PENDING_REMOVAL]),
}

def check_service_exists(method_name : str, database : Database, context_id : str, service_id : str):
    db_context = database.context(context_id).create()
    service_exists = db_context.services.contains(service_id)
    if method_name in ['CreateService']:
        if not service_exists: return
        msg = 'Context({})/Service({}) already exists in the database.'
        msg = msg.format(context_id, service_id)
        raise ServiceException(grpc.StatusCode.ALREADY_EXISTS, msg)
    elif method_name in ['UpdateService', 'DeleteService', 'GetServiceById']:
        if service_exists: return
        msg = 'Context({})/Service({}) does not exist in the database.'
        msg = msg.format(context_id, service_id)
        raise ServiceException(grpc.StatusCode.NOT_FOUND, msg)
    else:                                       # pragma: no cover (test requires malforming the code)
        msg = ' '.join([
            'Unexpected condition [check_service_exists(',
            'method_name={}, context_id={}, service_id={}, service_exists={})]',
        ])
        msg = msg.format(str(method_name), str(database), str(context_id), str(service_id), str(service_exists))
        raise ServiceException(grpc.StatusCode.UNIMPLEMENTED, msg)

def _check_enum(enum_name, method_name, value, to_enum_method, accepted_values_dict) -> Enum:
    _value = to_enum_method(value)
    if _value is None:                          # pragma: no cover (gRPC prevents unsupported values)
        msg = 'Unsupported {}({}).'
        msg = msg.format(enum_name, value)
        raise ServiceException(grpc.StatusCode.INVALID_ARGUMENT, msg)

    accepted_values = accepted_values_dict.get(method_name)
    if accepted_values is None:                 # pragma: no cover (test requires malforming the code)
        msg = '{} acceptable values not specified for Method({}).'
        msg = msg.format(enum_name, method_name)
        raise ServiceException(grpc.StatusCode.INTERNAL, msg)

    if len(accepted_values) == 0: return _value
    if _value in accepted_values: return _value

    msg = 'Method({}) does not accept {}({}). Permitted values for Method({}) are {}({}).'
    accepted_values_list = sorted(map(lambda v: v.name, accepted_values))
    msg = msg.format(method_name, enum_name, _value.name, method_name, enum_name, accepted_values_list)
    raise ServiceException(grpc.StatusCode.INVALID_ARGUMENT, msg)

def check_service_type(method_name : str, value : str) -> ServiceType:
    return _check_enum('ServiceType', method_name, value, to_servicetype_enum, ACCEPTED_SERVICE_TYPES)

def check_service_state(method_name : str, value : str) -> ServiceState:
    return _check_enum('ServiceState', method_name, value, to_servicestate_enum, ACCEPTED_SERVICE_STATES)

def check_service_constraint(
    logger : logging.Logger, constraint_number : int, parent_name : str, constraint : Constraint,
    add_constraints : Dict[str, Dict[str, Set[str]]]) -> Tuple[str, str]:

    try:
        constraint_type  = chk_string('constraint[#{}].constraint_type'.format(constraint_number),
                                    constraint.constraint_type,
                                    allow_empty=False)
        constraint_value = chk_string('constraint[#{}].constraint_value'.format(constraint_number),
                                    constraint.constraint_value,
                                    allow_empty=False)
    except Exception as e:
        logger.exception('Invalid arguments:')
        raise ServiceException(grpc.StatusCode.INVALID_ARGUMENT, str(e))

    if constraint_type in add_constraints:
        msg = 'Duplicated ConstraintType({}) in {}.'
        msg = msg.format(constraint_type, parent_name)
        raise ServiceException(grpc.StatusCode.INVALID_ARGUMENT, msg)

    add_constraints[constraint_type] = constraint_value
    return constraint_type, constraint_value

def check_service_endpoint(
    logger : logging.Logger, endpoint_number : int, parent_name : str, endpoint : EndPointId, service_context_id : str,
    add_topology_devices_endpoints : Dict[str, Dict[str, Set[str]]]) -> Tuple[str, str, str]:

    try:
        ep_context_id  = chk_string('endpoint[#{}].topoId.contextId.contextUuid.uuid'.format(endpoint_number),
                                    endpoint.topoId.contextId.contextUuid.uuid,
                                    allow_empty=True)
        ep_topology_id = chk_string('endpoint[#{}].topoId.topoId.uuid'.format(endpoint_number),
                                    endpoint.topoId.topoId.uuid,
                                    allow_empty=True)
        ep_device_id   = chk_string('endpoint[#{}].dev_id.device_id.uuid'.format(endpoint_number),
                                    endpoint.dev_id.device_id.uuid,
                                    allow_empty=False)
        ep_port_id     = chk_string('endpoint[#{}].port_id.uuid'.format(endpoint_number),
                                    endpoint.port_id.uuid,
                                    allow_empty=False)
    except Exception as e:
        logger.exception('Invalid arguments:')
        raise ServiceException(grpc.StatusCode.INVALID_ARGUMENT, str(e))

    if len(ep_context_id) == 0:
        # Assumption: if no context is specified for a service endpoint, use service context
        ep_context_id = service_context_id
    elif ep_context_id != service_context_id:
        # Assumption: service and service endpoints should belong to same context
        msg = 'Context({}) in {} mismatches service Context.'
        msg = msg.format(ep_context_id, parent_name)
        raise ServiceException(grpc.StatusCode.INVALID_ARGUMENT, msg)

    if len(ep_topology_id) == 0:
        # Assumption: if no topology is specified for a service endpoint, use default topology
        ep_topology_id = DEFAULT_TOPOLOGY_ID

    add_devices = add_topology_devices_endpoints.setdefault(ep_topology_id, dict())
    if ep_device_id in add_devices:
        msg = 'Duplicated Context({})/Topology({})/Device({}) in {}.'
        msg = msg.format(ep_context_id, ep_topology_id, ep_device_id, parent_name)
        raise ServiceException(grpc.StatusCode.INVALID_ARGUMENT, msg)

    add_device_and_endpoints = add_devices.setdefault(ep_device_id, set())

    # Implicit validation: same device cannot appear 2 times in the service
    #if ep_port_id in add_device_and_endpoints:
    #    msg = 'Duplicated Context({})/Topology({})/Device({})/Port({}) in {}.'
    #    msg = msg.format(ep_context_id, ep_topology_id, ep_device_id, ep_port_id, parent_name)
    #    raise ServiceException(grpc.StatusCode.NOT_FOUND, msg)

    add_device_and_endpoints.add(ep_port_id)
    return ep_topology_id, ep_device_id, ep_port_id

def check_device_endpoint_exists(
    database : Database, parent_name : str,
    context_id : str, topology_id : str, device_id : str, port_id : str) -> Endpoint:

    # Implicit validation: service.context == endpoint.context, and service.context created automatically
    if not database.contexts.contains(context_id):          # pragma: no cover
        msg = 'Context({}) in {} does not exist in the database.'
        msg = msg.format(context_id, parent_name)
        raise ServiceException(grpc.StatusCode.NOT_FOUND, msg)
    db_context = database.context(context_id)

    print('db_context.topologies', str(db_context.topologies.get()))

    if not db_context.topologies.contains(topology_id):
        msg = 'Context({})/Topology({}) in {} does not exist in the database.'
        msg = msg.format(context_id, topology_id, parent_name)
        raise ServiceException(grpc.StatusCode.NOT_FOUND, msg)
    db_topology = db_context.topology(topology_id)

    if not db_topology.devices.contains(device_id):
        msg = 'Context({})/Topology({})/Device({}) in {} does not exist in the database.'
        msg = msg.format(context_id, topology_id, device_id, parent_name)
        raise ServiceException(grpc.StatusCode.NOT_FOUND, msg)
    db_device = db_topology.device(device_id)

    if not db_device.endpoints.contains(port_id):
        msg = 'Context({})/Topology({})/Device({})/Port({}) in {} does not exist in the database.'
        msg = msg.format(context_id, topology_id, device_id, port_id, parent_name)
        raise ServiceException(grpc.StatusCode.NOT_FOUND, msg)

    return db_device.endpoint(port_id)

def check_service_request(
    method_name : str, request : Service, database : Database, logger : logging.Logger
    ) -> Tuple[str, str, ServiceType, str, ServiceState, List[Endpoint], List[Tuple[str, str]]]:

    # ----- Parse attributes -------------------------------------------------------------------------------------------
    try:
        context_id     = chk_string ('service.cs_id.contextId.contextUuid.uuid',
                                    request.cs_id.contextId.contextUuid.uuid,
                                    allow_empty=False)
        service_id     = chk_string ('service.cs_id.cs_id.uuid',
                                    request.cs_id.cs_id.uuid,
                                    allow_empty=False)
        service_type   = chk_options('service.serviceType',
                                    request.serviceType,
                                    servicetype_enum_values())
        service_config = chk_string ('service.serviceConfig.serviceConfig',
                                    request.serviceConfig.serviceConfig,
                                    allow_empty=True)
        service_state  = chk_options('service.serviceState.serviceState',
                                    request.serviceState.serviceState,
                                    servicestate_enum_values())
    except Exception as e:
        logger.exception('Invalid arguments:')
        raise ServiceException(grpc.StatusCode.INVALID_ARGUMENT, str(e))

    service_type = check_service_type(method_name, service_type)
    service_state = check_service_state(method_name, service_state)

    # ----- Check if service exists in database ------------------------------------------------------------------------
    check_service_exists(method_name, database, context_id, service_id)

    # ----- Parse constraints ------------------------------------------------------------------------------------------
    add_constraints : Dict[str, str] = {}
    constraint_tuples : List[Tuple[str, str]] = []
    for constraint_number,constraint in enumerate(request.constraint):
        parent_name = 'Constraint(#{}) of Context({})/Service({})'.format(constraint_number, context_id, service_id)
        constraint_type, constraint_value = check_service_constraint(
            logger, constraint_number, parent_name, constraint, add_constraints)
        constraint_tuples.append((constraint_type, constraint_value))

    # ----- Parse endpoints and check if they exist in the database ad device endpoints --------------------------------
    add_topology_devices_endpoints : Dict[str, Dict[str, Set[str]]] = {}
    db_endpoints : List[Endpoint] = []
    for endpoint_number,endpoint in enumerate(request.endpointList):
        parent_name = 'Endpoint(#{}) of Context({})/Service({})'.format(endpoint_number, context_id, service_id)

        ep_topology_id, ep_device_id, ep_port_id = check_service_endpoint(
            logger, endpoint_number, parent_name, endpoint, context_id, add_topology_devices_endpoints)

        db_endpoint = check_device_endpoint_exists(
            database, parent_name, context_id, ep_topology_id, ep_device_id, ep_port_id)
        db_endpoints.append(db_endpoint)

    return context_id, service_id, service_type, service_config, service_state, db_endpoints, constraint_tuples

def check_service_id_request(
    method_name : str, request : ServiceId, database : Database, logger : logging.Logger) -> Tuple[str, str]:

    # ----- Parse attributes -------------------------------------------------------------------------------------------
    try:
        context_id     = chk_string ('service_id.contextId.contextUuid.uuid',
                                    request.contextId.contextUuid.uuid,
                                    allow_empty=False)
        service_id     = chk_string ('service_id.cs_id.uuid',
                                    request.cs_id.uuid,
                                    allow_empty=False)
    except Exception as e:
        logger.exception('Invalid arguments:')
        raise ServiceException(grpc.StatusCode.INVALID_ARGUMENT, str(e))

    # ----- Check if service exists in database ------------------------------------------------------------------------
    check_service_exists(method_name, database, context_id, service_id)

    return context_id, service_id
