# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import logging
from queue import Queue
from random import random

import pytz
from apscheduler.schedulers.background import BackgroundScheduler

from common.proto.monitoring_pb2 import Kpi, KpiList
from common.tools.timestamp import Converters
from common.tools.timestamp.Converters import timestamp_utcnow_to_float, timestamp_float_to_string
from datetime import datetime
import time

from monitoring.service import MetricsDBTools
from monitoring.service.ManagementDBTools import ManagementDB
from monitoring.service.MetricsDBTools import MetricsDB

LOGGER = logging.getLogger(__name__)

class SubscriptionManager():
    def __init__(self, metrics_db):
        self.metrics_db = metrics_db
        self.scheduler = BackgroundScheduler(executors={'processpool': ProcessPoolExecutor(max_workers=20)})
        self.scheduler.start()
        
    def create_subscription(self,subs_queue, subscription_id, kpi_id, sampling_interval_s, sampling_duration_s=None, start_timestamp=None, end_timestamp=None):
        start_date = None
        end_date = None
        if sampling_duration_s:
            if not start_timestamp:
                start_timestamp = time.time()
            end_timestamp = start_timestamp + sampling_duration_s
        if start_timestamp:
            start_date = datetime.utcfromtimestamp(start_timestamp).isoformat()
        if end_timestamp:
            end_date = datetime.utcfromtimestamp(end_timestamp).isoformat()

        LOGGER.debug(f"kpi_id: {kpi_id}")
        LOGGER.debug(f"sampling_interval_s: {sampling_interval_s}")
        LOGGER.debug(f"subscription_id: {subscription_id}")
        LOGGER.debug(f"start_date: {start_date}")
        self.scheduler.add_job(self.metrics_db.get_subscription_data, args=(subs_queue,kpi_id, sampling_interval_s),
                               trigger='interval', seconds=sampling_interval_s, start_date=start_date,
                               end_date=end_date, timezone=pytz.utc, id=str(subscription_id))
        LOGGER.debug(f"Subscrition job {subscription_id} succesfully created")

    def delete_subscription(self, subscription_id):
        self.scheduler.remove_job(subscription_id)