# Copyright 2022-2024 ETSI OSG/SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
import logging
from typing import Any, Dict, Optional, Tuple, TypedDict

import requests

from common.proto.kpi_sample_types_pb2 import KpiSampleType
from common.type_checkers.Checkers import chk_attribute, chk_string, chk_type
from device.service.driver_api._Driver import RESOURCE_ENDPOINTS

from .Constants import SPECIAL_RESOURCE_MAPPINGS


class LANPrefixesDict(TypedDict):
    lan: str
    lan_tag: str


LOGGER = logging.getLogger(__name__)

SITE_NETWORK_ACCESS_TYPE = "ietf-l3vpn-svc:multipoint"


def service_exists(wim_url: str, auth, service_uuid: str) -> bool:
    try:
        get_connectivity_service(wim_url, auth, service_uuid)
        return True
    except:  # pylint: disable=bare-except
        return False


def get_all_active_connectivity_services(wim_url: str, auth):
    try:
        LOGGER.info("Sending get all connectivity services")
        servicepoint = f"{wim_url}/restconf/data/ietf-l3vpn-svc:l3vpn-svc/vpn-services"
        response = requests.get(servicepoint, auth=auth)

        if response.status_code != requests.codes.ok:
            raise Exception(
                "Unable to get all connectivity services",
                http_code=response.status_code,
            )

        return response
    except requests.exceptions.ConnectionError:
        raise Exception("Request Timeout", http_code=408)


def get_connectivity_service(wim_url, auth, service_uuid):
    try:
        LOGGER.info("Sending get connectivity service")
        servicepoint = f"{wim_url}/restconf/data/ietf-l3vpn-svc:l3vpn-svc/vpn-service={service_uuid}"

        response = requests.get(servicepoint, auth=auth)

        if response.status_code != requests.codes.ok:
            raise Exception(
                "Unable to get connectivity service{:s}".format(str(service_uuid)),
                http_code=response.status_code,
            )

        return response
    except requests.exceptions.ConnectionError:
        raise Exception("Request Timeout", http_code=408)


def create_l3vpn_datamodel(service_uuid, resource_value: dict) -> dict:
    src_device_uuid: str = resource_value["src_device_name"]
    src_endpoint_uuid: str = resource_value["src_endpoint_name"]
    src_site_location: str = resource_value["src_site_location"]
    src_ipv4_lan_prefixes: list[LANPrefixesDict] = resource_value.get(
        "src_ipv4_lan_prefixes"
    )
    src_site_id: str = resource_value.get("src_site_id", f"site_{src_site_location}")
    src_management_type: str = resource_value.get(
        "src_management_type", "ietf-l3vpn-svc:customer-managed"
    )
    if src_management_type != "ietf-l3vpn-svc:customer-managed":
        raise Exception("management type %s not supported", src_management_type)
    src_role: str = "ietf-l3vpn-svc:hub-role"
    src_ce_address: str = resource_value["src_ce_address"]
    src_pe_address: str = resource_value["src_pe_address"]
    src_ce_pe_network_prefix: int = resource_value["src_ce_pe_network_prefix"]
    src_mtu: int = resource_value["src_mtu"]
    src_input_bw: int = resource_value.get("src_input_bw", 1000000000)
    src_output_bw: int = resource_value.get("src_input_bw", 1000000000)
    src_qos_profile_id: str = resource_value.get(
        "src_qos_profile_id", "src_qos_profile"
    )
    src_qos_profile_direction: str = (
        resource_value.get("src_qos_profile_direction", "ietf-l3vpn-svc:both"),
    )
    src_qos_profile_latency: int = resource_value.get("src_qos_profile_latency", 10)
    src_qos_profile_bw_guarantee: int = resource_value.get(
        "src_qos_profile_bw_guarantee", 100
    )

    dst_device_uuid = resource_value["dst_device_name"]
    dst_endpoint_uuid = resource_value["dst_endpoint_name"]
    dst_site_location: str = resource_value["dst_site_location"]
    dst_ipv4_lan_prefixes: list[LANPrefixesDict] = resource_value[
        "dst_ipv4_lan_prefixes"
    ]
    dst_site_id: str = resource_value.get("dst_site_id", f"site_{dst_site_location}")
    dst_management_type: str = resource_value.get(
        "dst_management_type", "ietf-l3vpn-svc:customer-managed"
    )
    if dst_management_type != "ietf-l3vpn-svc:customer-managed":
        raise Exception("management type %s not supported", dst_management_type)
    dst_role: str = "ietf-l3vpn-svc:spoke-role"
    dst_ce_address: str = resource_value["dst_ce_address"]
    dst_pe_address: str = resource_value["dst_pe_address"]
    dst_ce_pe_network_prefix: int = resource_value["dst_ce_pe_network_prefix"]
    dst_mtu: int = resource_value["dst_mtu"]
    dst_input_bw: int = resource_value.get("dst_input_bw", 1000000000)
    dst_output_bw: int = resource_value.get("dst_output_bw", 1000000000)
    dst_qos_profile_id: str = resource_value.get(
        "dst_qos_profile_id", "dst_qos_profile"
    )
    dst_qos_profile_direction: str = (
        resource_value.get("dst_qos_profile_direction", "ietf-l3vpn-svc:both"),
    )
    dst_qos_profile_latency: int = resource_value.get("dst_qos_profile_latency", 10)
    dst_qos_profile_bw_guarantee: int = resource_value.get(
        "dst_qos_profile_bw_guarantee", 100
    )

    # Create source site information
    src_management = {"type": src_management_type}
    src_locations = {"location": [{"location-id": src_site_location}]}
    src_devices = {
        "device": [{"device-id": src_device_uuid, "location": src_site_location}]
    }
    src_site_lan_prefixes = [
        {"lan": lp["lan"], "lan-tag": lp["lan_tag"], "next-hop": src_pe_address}
        for lp in src_ipv4_lan_prefixes
    ]
    src_site_routing_protocols = {
        "routing-protocol": [
            {
                "type": "ietf-l3vpn-svc:static",
                "static": {
                    "cascaded-lan-prefixes": {
                        "ipv4-lan-prefixes": src_site_lan_prefixes
                    }
                },
            }
        ]
    }
    src_site_network_accesses = {
        "site-network-access": [
            {
                "site-network-access-id": src_endpoint_uuid,
                "site-network-access-type": SITE_NETWORK_ACCESS_TYPE,
                "device-reference": src_device_uuid,
                "vpn-attachment": {"vpn-id": service_uuid, "site-role": src_role},
                "ip-connection": {
                    "ipv4": {
                        "address-allocation-type": "ietf-l3vpn-svc:static-address",
                        "addresses": {
                            "provider-address": src_pe_address,
                            "customer-address": src_ce_address,
                            "prefix-length": src_ce_pe_network_prefix,
                        },
                    }
                },
                "service": {
                    "svc-mtu": src_mtu,
                    "svc-input-bandwidth": src_input_bw,
                    "svc-output-bandwidth": src_output_bw,
                    "qos": {
                        "qos-profile": {
                            "classes": {
                                "class": [
                                    {
                                        "class-id": src_qos_profile_id,
                                        "direction": src_qos_profile_direction,
                                        "latency": {
                                            "latency-boundary": src_qos_profile_latency
                                        },
                                        "bandwidth": {
                                            "guaranteed-bw-percent": src_qos_profile_bw_guarantee
                                        },
                                    }
                                ]
                            }
                        }
                    },
                },
            }
        ]
    }

    # Create destination site information
    dst_management = {"type": src_management_type}
    dst_locations = {"location": [{"location-id": dst_site_location}]}
    dst_devices = {
        "device": [{"device-id": dst_device_uuid, "location": dst_site_location}]
    }
    dst_site_lan_prefixes = [
        {"lan": lp["lan"], "lan-tag": lp["lan_tag"], "next-hop": dst_pe_address}
        for lp in dst_ipv4_lan_prefixes
    ]
    dst_site_routing_protocols = {
        "routing-protocol": [
            {
                "type": "ietf-l3vpn-svc:static",
                "static": {
                    "cascaded-lan-prefixes": {
                        "ipv4-lan-prefixes": dst_site_lan_prefixes
                    }
                },
            }
        ]
    }
    dst_site_network_accesses = {
        "site-network-access": [
            {
                "site-network-access-id": dst_endpoint_uuid,
                "site-network-access-type": SITE_NETWORK_ACCESS_TYPE,
                "device-reference": dst_device_uuid,
                "vpn-attachment": {"vpn-id": service_uuid, "site-role": dst_role},
                "ip-connection": {
                    "ipv4": {
                        "address-allocation-type": "ietf-l3vpn-svc:static-address",
                        "addresses": {
                            "provider-address": dst_pe_address,
                            "customer-address": dst_ce_address,
                            "prefix-length": dst_ce_pe_network_prefix,
                        },
                    }
                },
                "service": {
                    "svc-mtu": dst_mtu,
                    "svc-input-bandwidth": dst_input_bw,
                    "svc-output-bandwidth": dst_output_bw,
                    "qos": {
                        "qos-profile": {
                            "classes": {
                                "class": [
                                    {
                                        "class-id": dst_qos_profile_id,
                                        "direction": dst_qos_profile_direction,
                                        "latency": {
                                            "latency-boundary": dst_qos_profile_latency
                                        },
                                        "bandwidth": {
                                            "guaranteed-bw-percent": dst_qos_profile_bw_guarantee
                                        },
                                    }
                                ]
                            }
                        }
                    },
                },
            }
        ]
    }

    sites = {
        "site": [
            {
                "site-id": src_site_id,
                "management": src_management,
                "locations": src_locations,
                "devices": src_devices,
                "routing-protocols": src_site_routing_protocols,
                "site-network-accesses": src_site_network_accesses,
            },
            {
                "site-id": dst_site_id,
                "management": dst_management,
                "locations": dst_locations,
                "devices": dst_devices,
                "routing-protocols": dst_site_routing_protocols,
                "site-network-accesses": dst_site_network_accesses,
            },
        ]
    }

    l3_vpn_data_model = {
        "ietf-l3vpn-svc:l3vpn-svc": {
            "vpn-services": {"vpn-service": [{"vpn-id": service_uuid}]},
            "sites": sites,
        }
    }

    return l3_vpn_data_model


def process_optional_string_field(
    endpoint_data: Dict[str, Any],
    field_name: str,
    endpoint_resource_value: Dict[str, Any],
) -> None:
    field_value = chk_attribute(
        field_name, endpoint_data, "endpoint_data", default=None
    )
    if field_value is None:
        return
    chk_string("endpoint_data.{:s}".format(field_name), field_value)
    if len(field_value) > 0:
        endpoint_resource_value[field_name] = field_value


def compose_resource_endpoint(
    endpoint_data: Dict[str, Any],
) -> Optional[Tuple[str, Dict]]:
    try:
        # Check type of endpoint_data
        chk_type("endpoint_data", endpoint_data, dict)

        # Check endpoint UUID (mandatory)
        endpoint_uuid = chk_attribute("uuid", endpoint_data, "endpoint_data")
        chk_string("endpoint_data.uuid", endpoint_uuid, min_length=1)
        endpoint_resource_path = SPECIAL_RESOURCE_MAPPINGS.get(RESOURCE_ENDPOINTS)
        endpoint_resource_key = "{:s}/endpoint[{:s}]".format(
            endpoint_resource_path, endpoint_uuid
        )
        endpoint_resource_value = {"uuid": endpoint_uuid}

        # Check endpoint optional string fields
        process_optional_string_field(endpoint_data, "name", endpoint_resource_value)
        process_optional_string_field(endpoint_data, "type", endpoint_resource_value)
        process_optional_string_field(
            endpoint_data, "context_uuid", endpoint_resource_value
        )
        process_optional_string_field(
            endpoint_data, "topology_uuid", endpoint_resource_value
        )

        # Check endpoint sample types (optional)
        endpoint_sample_types = chk_attribute(
            "sample_types", endpoint_data, "endpoint_data", default=[]
        )
        chk_type("endpoint_data.sample_types", endpoint_sample_types, list)
        sample_types = {}
        sample_type_errors = []
        for i, endpoint_sample_type in enumerate(endpoint_sample_types):
            field_name = "endpoint_data.sample_types[{:d}]".format(i)
            try:
                chk_type(field_name, endpoint_sample_type, (int, str))
                if isinstance(endpoint_sample_type, int):
                    metric_name = KpiSampleType.Name(endpoint_sample_type)
                    metric_id = endpoint_sample_type
                elif isinstance(endpoint_sample_type, str):
                    metric_id = KpiSampleType.Value(endpoint_sample_type)
                    metric_name = endpoint_sample_type
                else:
                    str_type = str(type(endpoint_sample_type))
                    raise Exception("Bad format: {:s}".format(str_type))  # pylint: disable=broad-exception-raised
            except Exception as e:  # pylint: disable=broad-exception-caught
                MSG = "Unsupported {:s}({:s}) : {:s}"
                sample_type_errors.append(
                    MSG.format(field_name, str(endpoint_sample_type), str(e))
                )

            metric_name = metric_name.lower().replace("kpisampletype_", "")
            monitoring_resource_key = "{:s}/state/{:s}".format(
                endpoint_resource_key, metric_name
            )
            sample_types[metric_id] = monitoring_resource_key

        if len(sample_type_errors) > 0:
            # pylint: disable=broad-exception-raised
            raise Exception(
                "Malformed Sample Types:\n{:s}".format("\n".join(sample_type_errors))
            )

        if len(sample_types) > 0:
            endpoint_resource_value["sample_types"] = sample_types

        if "location" in endpoint_data:
            endpoint_resource_value["location"] = endpoint_data["location"]

        if "ce-ip" in endpoint_data:
            endpoint_resource_value["ce-ip"] = endpoint_data["ce-ip"]

        if "address_ip" in endpoint_data:
            endpoint_resource_value["address_ip"] = endpoint_data["address_ip"]

        if "address_prefix" in endpoint_data:
            endpoint_resource_value["address_prefix"] = endpoint_data["address_prefix"]

        if "mtu" in endpoint_data:
            endpoint_resource_value["mtu"] = endpoint_data["mtu"]

        if "ipv4_lan_prefixes" in endpoint_data:
            endpoint_resource_value["ipv4_lan_prefixes"] = endpoint_data[
                "ipv4_lan_prefixes"
            ]

        return endpoint_resource_key, endpoint_resource_value
    except:  # pylint: disable=bare-except
        LOGGER.exception("Problem composing endpoint({:s})".format(str(endpoint_data)))
        return None
