# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import logging
import signal
import sys
import threading

from common.Constants import DEFAULT_GRPC_GRACE_PERIOD, DEFAULT_GRPC_MAX_WORKERS
from common.Settings import get_log_level, get_metrics_port, get_setting
from dbscanserving.Config import GRPC_SERVICE_PORT
from dbscanserving.service.DbscanService import DbscanService
from prometheus_client import start_http_server

terminate = threading.Event()
LOGGER = None


def signal_handler(signal, frame):  # pylint: disable=redefined-outer-name
    LOGGER.warning("Terminate signal received")
    terminate.set()


def main():
    global LOGGER  # pylint: disable=global-statement

    log_level = get_log_level()
    logging.basicConfig(level=log_level)
    LOGGER = logging.getLogger(__name__)

    service_port = get_setting(
        "DBSCANSERVICE_SERVICE_PORT_GRPC", default=GRPC_SERVICE_PORT
    )
    grace_period = get_setting("GRACE_PERIOD", default=DEFAULT_GRPC_GRACE_PERIOD)
    max_workers = get_setting("MAX_WORKERS", default=DEFAULT_GRPC_MAX_WORKERS)

    signal.signal(signal.SIGINT, signal_handler)
    signal.signal(signal.SIGTERM, signal_handler)

    LOGGER.info("Starting...")

    # Start metrics server
    metrics_port = get_metrics_port()
    start_http_server(metrics_port)

    # Starting CentralizedCybersecurity service
    grpc_service = DbscanService(
        port=service_port, max_workers=max_workers, grace_period=grace_period
    )
    grpc_service.start()

    # Wait for Ctrl+C or termination signal
    while not terminate.wait(timeout=0.1):
        pass

    LOGGER.info("Terminating...")
    grpc_service.stop()

    LOGGER.info("Bye")
    return 0


if __name__ == "__main__":
    sys.exit(main())
