# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import logging, signal, sys, threading
from prometheus_client import start_http_server
from common.Settings import get_setting, wait_for_environment_variables
from context.client.ContextClient import ContextClient
from interdomain.client.InterdomainClient import InterdomainClient
from service.client.ServiceClient import ServiceClient
from slice.Config import (
    CONTEXT_SERVICE_HOST, CONTEXT_SERVICE_PORT, GRPC_SERVICE_PORT, GRPC_MAX_WORKERS, GRPC_GRACE_PERIOD,
    INTERDOMAIN_SERVICE_HOST, INTERDOMAIN_SERVICE_PORT, LOG_LEVEL, METRICS_PORT, SERVICE_SERVICE_HOST,
    SERVICE_SERVICE_PORT)
from .SliceService import SliceService

terminate = threading.Event()
LOGGER : logging.Logger = None

def signal_handler(signal, frame): # pylint: disable=redefined-outer-name
    LOGGER.warning('Terminate signal received')
    terminate.set()

def main():
    global LOGGER # pylint: disable=global-statement

    grpc_service_port        = get_setting('SLICESERVICE_SERVICE_PORT_GRPC',       default=GRPC_SERVICE_PORT       )
    max_workers              = get_setting('MAX_WORKERS',                          default=GRPC_MAX_WORKERS        )
    grace_period             = get_setting('GRACE_PERIOD',                         default=GRPC_GRACE_PERIOD       )
    log_level                = get_setting('LOG_LEVEL',                            default=LOG_LEVEL               )
    metrics_port             = get_setting('METRICS_PORT',                         default=METRICS_PORT            )

    logging.basicConfig(level=log_level)
    LOGGER = logging.getLogger(__name__)

    wait_for_environment_variables([
        'CONTEXTSERVICE_SERVICE_HOST', 'CONTEXTSERVICE_SERVICE_PORT_GRPC',
        'INTERDOMAINSERVICE_SERVICE_HOST', 'INTERDOMAINSERVICE_SERVICE_PORT_GRPC',
        'SERVICESERVICE_SERVICE_HOST', 'SERVICESERVICE_SERVICE_PORT_GRPC',
    ])

    context_service_host     = get_setting('CONTEXTSERVICE_SERVICE_HOST',          default=CONTEXT_SERVICE_HOST    )
    context_service_port     = get_setting('CONTEXTSERVICE_SERVICE_PORT_GRPC',     default=CONTEXT_SERVICE_PORT    )
    interdomain_service_host = get_setting('INTERDOMAINSERVICE_SERVICE_HOST',      default=INTERDOMAIN_SERVICE_HOST)
    interdomain_service_port = get_setting('INTERDOMAINSERVICE_SERVICE_PORT_GRPC', default=INTERDOMAIN_SERVICE_PORT)
    service_service_host     = get_setting('SERVICESERVICE_SERVICE_HOST',          default=SERVICE_SERVICE_HOST    )
    service_service_port     = get_setting('SERVICESERVICE_SERVICE_PORT_GRPC',     default=SERVICE_SERVICE_PORT    )

    signal.signal(signal.SIGINT,  signal_handler)
    signal.signal(signal.SIGTERM, signal_handler)

    LOGGER.info('Starting...')

    # Start metrics server
    start_http_server(metrics_port)

    # Initialize Context Client
    if context_service_host is None or context_service_port is None:
        raise Exception('Wrong address({:s}):port({:s}) of Context component'.format(
            str(context_service_host), str(context_service_port)))
    context_client = ContextClient(context_service_host, context_service_port)

    # Initialize Interdomain Client
    if interdomain_service_host is None or interdomain_service_port is None:
        raise Exception('Wrong address({:s}):port({:s}) of Interdomain component'.format(
            str(interdomain_service_host), str(interdomain_service_port)))
    interdomain_client = InterdomainClient(interdomain_service_host, interdomain_service_port)

    # Initialize Service Client
    if service_service_host is None or service_service_port is None:
        raise Exception('Wrong address({:s}):port({:s}) of Service component'.format(
            str(service_service_host), str(service_service_port)))
    service_client = ServiceClient(service_service_host, service_service_port)

    # Starting slice service
    grpc_service = SliceService(
        context_client, interdomain_client, service_client, port=grpc_service_port, max_workers=max_workers,
        grace_period=grace_period)
    grpc_service.start()

    # Wait for Ctrl+C or termination signal
    while not terminate.wait(timeout=0.1): pass

    LOGGER.info('Terminating...')
    grpc_service.stop()

    LOGGER.info('Bye')
    return 0

if __name__ == '__main__':
    sys.exit(main())
