package fabric

import com.google.protobuf.util.JsonFormat
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.json.Json
import org.hyperledger.fabric.gateway.Contract
import org.hyperledger.fabric.gateway.Wallet
import org.hyperledger.fabric.gateway.Wallets
import org.hyperledger.fabric.sdk.security.CryptoSuiteFactory
import org.hyperledger.fabric_ca.sdk.HFCAClient
import proto.Config
import java.nio.file.Paths
import java.util.*

class FabricConnector(val config: Config.DltConfig) {
    private val caClient: HFCAClient
    private val wallet: Wallet
    private val contract: Contract

    init {
        // Create a CA client for interacting with the CA.
        val props = Properties()
        props["pemFile"] = config.caCertFile
        props["allowAllHostNames"] = "true"
        caClient = HFCAClient.createNewInstance(config.caUrl, props)
        val cryptoSuite = CryptoSuiteFactory.getDefault().cryptoSuite
        caClient.cryptoSuite = cryptoSuite

        // Create a wallet for managing identities
        wallet = Wallets.newFileSystemWallet(Paths.get(config.wallet))
        contract = connect()
    }

    fun connect(): Contract {
        enrollAdmin(config, caClient, wallet)
        registerUser(config, caClient, wallet)
        return getContract(config, wallet)
    }

    fun putData(record: Config.DltRecord): String {
        println(record.type.toString())
        return String(
            contract.submitTransaction(
                "AddRecord",
                record.id.uuid,
                record.type.number.toString(),
                record.json
            )
        )
    }

    fun getData(uuid: String): Config.DltRecord {
        // TODO: get rid of json encoding/decoding
        val result = contract.evaluateTransaction("GetRecord", uuid)
        println(String(result))
        val builder = Config.DltRecord.newBuilder()
        val protoRes = JsonFormat.parser().merge(String(result), builder)
        return builder.build()
    }

    fun updateData(record: Config.DltRecord): String {
        return String(
            contract.submitTransaction(
                "UpdateRecord",
                record.id.uuid,
                record.type.number.toString(),
                record.json
            )
        )
    }

    fun deleteData(uuid: String): String {
        return String(contract.submitTransaction("DeactivateRecord", uuid))
    }
}