# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import logging, signal, sys, threading
from prometheus_client import start_http_server
from common.Settings import get_log_level, get_metrics_port, get_setting
from common.message_broker.Factory import get_messagebroker_backend
from common.message_broker.MessageBroker import MessageBroker
from context.Config import POPULATE_FAKE_DATA
from sqlalchemy.orm import sessionmaker, declarative_base
from context.service.database.Base import Base
from .grpc_server.ContextService import ContextService
from .rest_server.Resources import RESOURCES
from .rest_server.RestServer import RestServer
from .Populate import populate
# from models import Device, EndPoint, EndPointId, DeviceDriverEnum, DeviceOperationalStatusEnum, ConfigActionEnum, \
#     ConfigRule, KpiSampleType, Base
from sqlalchemy import create_engine

terminate = threading.Event()
LOGGER = None

def signal_handler(signal, frame): # pylint: disable=redefined-outer-name
    LOGGER.warning('Terminate signal received')
    terminate.set()

def main():
    global LOGGER # pylint: disable=global-statement

    log_level = get_log_level()
    logging.basicConfig(level=log_level)
    LOGGER = logging.getLogger(__name__)

    signal.signal(signal.SIGINT,  signal_handler)
    signal.signal(signal.SIGTERM, signal_handler)

    LOGGER.info('Starting...')

    # Start metrics server
    metrics_port = get_metrics_port()
    start_http_server(metrics_port)

    # Get database instance
    db_uri = 'cockroachdb://root@10.152.183.66:26257/defaultdb?sslmode=disable'
    LOGGER.debug('Connecting to DB: {}'.format(db_uri))

    # engine = create_engine(db_uri, echo=False)

    try:
        engine = create_engine(db_uri)
    except Exception as e:
        LOGGER.error("Failed to connect to database.")
        LOGGER.error(f"{e}")
        return 1

    Base.metadata.create_all(engine)
    session = sessionmaker(bind=engine)

    # Get message broker instance
    messagebroker = MessageBroker(get_messagebroker_backend())

    # Starting context service
    grpc_service = ContextService(session, messagebroker)
    grpc_service.start()

    rest_server = RestServer()
    for endpoint_name, resource_class, resource_url in RESOURCES:
        rest_server.add_resource(resource_class, resource_url, endpoint=endpoint_name, resource_class_args=(session,))
    rest_server.start()

    populate_fake_data = get_setting('POPULATE_FAKE_DATA', default=POPULATE_FAKE_DATA)
    if isinstance(populate_fake_data, str): populate_fake_data = (populate_fake_data.upper() in {'T', '1', 'TRUE'})
    if populate_fake_data:
        LOGGER.info('Populating fake data...')
        populate(host='127.0.0.1', port=grpc_service.bind_port)
        LOGGER.info('Fake Data populated')

    # Wait for Ctrl+C or termination signal
    while not terminate.wait(timeout=0.1): pass

    LOGGER.info('Terminating...')
    grpc_service.stop()
    rest_server.shutdown()
    rest_server.join()

    LOGGER.info('Bye')
    return 0

if __name__ == '__main__':
    sys.exit(main())
