/*
 * Copyright 2020 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref;

import java.awt.Desktop;
import java.awt.GraphicsEnvironment;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;

import javax.swing.SwingUtilities;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import fr.mines_stetienne.ci.saref.SAREFPipeline.Mode;
import fr.mines_stetienne.ci.saref.gui.MainFrame;
import fr.mines_stetienne.ci.saref.managers.RepositoryManager;
import fr.mines_stetienne.ci.saref.tests.TestSuites;

/**
 *
 * @author Maxime Lefrançois
 *
 */
public class Main {

	private static final Logger LOG = LoggerFactory.getLogger(Main.class);
	private static boolean CAN_BROWSE = false;
	static {
		try {
			CAN_BROWSE = Desktop.isDesktopSupported()
			&& Desktop.getDesktop().isSupported(Desktop.Action.BROWSE);
		} catch (Exception ex) {}
	}

	private static final String CI_SERVER_HOST = System.getenv("CI_SERVER_HOST");
	private static final String CI_PROJECT_PATH = System.getenv("CI_PROJECT_PATH");
	private static final String CI_PIPELINE_ID = System.getenv("CI_PIPELINE_ID");
	private static final boolean isGitLabCI = CI_SERVER_HOST != null && CI_PROJECT_PATH != null && CI_PIPELINE_ID != null;

	public static void main(String[] args) {
		if (args.length == 0 && !GraphicsEnvironment.isHeadless()) {
			mainGUI();
		} else {
			mainCLI(args);
		}
	}

	private static void mainCLI(String[] args) {
		CLIExecution cliExecution = new CLIExecution();
		final SAREFPipeline pipeline = cliExecution.getSAREFPipeline(args);
		TestSuites testSuites = pipeline.run();
		try {
			if (pipeline.mode != Mode.CLEAN) {
				System.out.println(String.format("\n\nThe SAREF Pipeline completed with %s errors and %s warnings\n",
						testSuites.getErrors(), testSuites.getFailures()));

				File report = new File(pipeline.targetDir, "report_output.xml");
				if (isGitLabCI) {
				} else {
					System.out.println("\n\n\nSAREF Pipeline XML report:\n" + report.getCanonicalPath());
				}

				File reportHTML = new File(pipeline.targetDir, "site/report.html");
				if (reportHTML.isFile()) {
					if (CAN_BROWSE)
						Desktop.getDesktop().browse(reportHTML.toURI());
					else if (isGitLabCI) {
						String link;
						if(pipeline.mode == Mode.DEVELOP || pipeline.mode == Mode.RELEASE) {
							link = String.format("%ssnapshot/%s/report.html", SAREF.BASE, CI_PIPELINE_ID);
						} else {
							link = String.format("%sstaging/%s/report.html", SAREF.BASE, CI_PIPELINE_ID);
						}
						System.out.println("\n\n\nSAREF pipeline HTML report:\n" + link);
					} else {
						System.out.println("\n\n\nSAREF pipeline HTML report:\n" + reportHTML.getCanonicalPath());
					}
				}
				if (!pipeline.ignoreSite) {
					RepositoryManager target = pipeline.getSourcesManager().getTargetRepositoryManager();
					if (target != null) {
						File siteDir = new File(pipeline.targetDir, SAREF.NAME_SITE);
						File ontoDir = new File(siteDir, target.getCurrentVersion().getVersionPath());
						File documentationHTML = new File(ontoDir,
								String.format("%s.html", target.getRepository().getOntologyFileName()));
						if (CAN_BROWSE) {
							Desktop.getDesktop().browse(documentationHTML.toURI());
						} else if (isGitLabCI) {
							String link = String.format("%ssnapshot/%s/%s/%s/%s.html", SAREF.BASE, CI_PIPELINE_ID, target.getRepository().getProject().getPath(),
										target.getCurrentVersionName(),
										target.getRepository().getProject().getOntologyFileName());
							System.out.println("\n\n\nHTML documentation:\n" + link);
						} else {
							System.out.println(String.format("\nDocumentation for %s %s: %s\n",
									target.getRepository().getProject(), target.getCurrentVersionName(),
									documentationHTML.getCanonicalPath()));
						}
					} else {
						String link = String.format("%sstaging/%s/", SAREF.BASE, CI_PIPELINE_ID);
						System.out.println("\n\n\nHTML documentation:\n" + link);
					}
				}
			}
		} catch (IOException ex) {
			LOG.error("Exception:", ex);
			System.exit(-1);
		}
		if (testSuites.getErrors() > 0) {
			System.exit(-1);
		}
	}

	private static void mainGUI() {
		try {
			String info = IOUtils.toString(Main.class.getClassLoader().getResourceAsStream("cli/info.txt"),
					StandardCharsets.UTF_8);
			System.out.println(info);
			MainFrame application = new MainFrame();
			SwingUtilities.updateComponentTreeUI(application);
			java.awt.EventQueue.invokeLater(new Runnable() {
				@Override
				public void run() {
					application.toFront();
					application.repaint();
				}
			});
		} catch (Exception ex) {
			LOG.warn("Exception ", ex);
			System.exit(-1);
		}
	}

}
