/*
 * Copyright 2024 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.managers;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URL;
import java.net.URLEncoder;
import java.net.http.*;
import java.nio.file.Files;
import java.nio.charset.StandardCharsets;
import java.nio.file.Paths;
import java.util.Map;
import java.util.List;
import java.util.HashMap;
import java.util.ArrayList;
//import org.apache.commons.lang3.StringEscapeUtils;
import com.fasterxml.jackson.databind.ObjectMapper;
import fr.mines_stetienne.ci.saref.SAREFPipelineException;

/*	docker run -d --name SAREFvalidator -p 8080:8080 isaitb/shacl-validator	 # runs the generic validator locally given a  Dockerfile.
	Docs for RDF validator: https://www.itb.ec.europa.eu/docs/guides/latest/validatingRDF/index.html
	Base docker image: https://hub.docker.com/r/isaitb/shacl-validator  (generic validator)
	Use Swagger UI for testing: http://localhost:8080/shacl/swagger-ui/index.html#/{domain}/api/validate or http://193.49.174.57:4050/shacl/swagger-ui/index.html#
	Online validator: https://www.itb.ec.europa.eu/shacl/any/upload
* */
public class ShaclValidationManager {

	private static String VALIDATOR_URL = "http://localhost:8080/shacl/"; //"http://193.49.174.57:4050/shacl/";
	private final String TEXT_TURTLE = "text/turtle";
	private final String JSON_FORMAT = "application/json";
	private final String STRING_FORMAT = "STRING";

	private String ontologyToValidate;	// can be file or URL.
	private List<String> shaclFileList;
	private List<RuleSet> ruleSetList;
	private HttpClient client;
	private ObjectMapper objectMapper;
	private ValidationRequest validationRequest;

	private static String encodeBody(Object obj) {
		return URLEncoder.encode(obj.toString(), StandardCharsets.UTF_8);
	}

	private enum MESSAGE {	// each of these accepts a single string parameter.
		http_exception, ioexception
	}

	class RuleSet {
		private String ruleSet;	// JSON-encoded_SHACL_shape

		public String getRuleSet() {
			return ruleSet;
		}
		public void setRuleSet(String ruleSet) {
			this.ruleSet = ruleSet;
		}
		// constants for now...
		public String getEmbeddingMethod()	{ return STRING_FORMAT; }
		public String getRuleSyntax() { return TEXT_TURTLE; }
	}

	class ValidationRequest {
		private String contentToValidate;    // JSON-encoded ontology
		private List<RuleSet> externalRules;

		public String getContentToValidate() {
			return contentToValidate;
		}

		public void setContentToValidate(String contentToValidate) {
			this.contentToValidate = contentToValidate;
		}

		public List<RuleSet> getExternalRules() {
			return externalRules;
		}

		public void setExternalRules(List<RuleSet> externalRules) {
			this.externalRules = externalRules;
		}

		// constants for now...
		public String getEmbeddingMethod() {
			return STRING_FORMAT;
		}

		public String getContentSyntax() {
			return TEXT_TURTLE;
		}

		public String getReportSyntax() {
			return TEXT_TURTLE;
		}

		public String getRdfReportSyntax() {
			return TEXT_TURTLE;
		}

		public String getLocale() {
			return "en";
		}

		// TODO: set loadImports to true to allow recursive searches.
		public boolean getLoadImports() {
			return false;
		}
	}

	/**
	 * Open a REST interface to the remote SHACL validator.
	 */
	public ShaclValidationManager(String ontologyToValidate) throws IOException {
		this.client = HttpClient.newHttpClient();
		this.objectMapper = new ObjectMapper();
		this.ontologyToValidate = ontologyToValidate;
		String url = System.getenv("VALIDATOR_URL");
		if (url != null)  VALIDATOR_URL = url;
		System.out.println("SHACL validation server running at "+VALIDATOR_URL);
	}

	/**
	 * Is the validation server accessible?
	 * @throws SAREFPipelineException
	 */
	public static void PingValidationServer() throws SAREFPipelineException {
		try {
			URL url = new URL(VALIDATOR_URL);
			HttpURLConnection connection = (HttpURLConnection) url.openConnection();
			connection.setRequestMethod("GET");
			connection.connect();
			connection.getResponseCode();
		} catch (IOException e) {
			throw new SAREFPipelineException("response", "Unknown host: " + VALIDATOR_URL);
		}
	}

	/**
	 * Read shacl files. Assign this.shaclFileList, this.ruleSetList.
	 * Preferred validation format is JSON-escaped text. Requires \r\n as line breaks!!!
	 */
	private void createRuleSetList(List<String> shaclFileList)  {
		this.shaclFileList = new ArrayList();
		this.ruleSetList = new ArrayList();

		for (int i = 0; i < shaclFileList.size(); i++) {
			String readingFile = shaclFileList.get(i);
			try {
				String fileStr = new String(Files.readAllBytes(Paths.get(readingFile)));
				RuleSet ruleSet = new RuleSet();
				ruleSet.setRuleSet(fileStr);
				this.shaclFileList.add(readingFile);
				this.ruleSetList.add(ruleSet);
			} catch (IOException ex) {
				System.out.println("ShaclValidationManager unable to read SHACL file: " + readingFile);
			}
		}
	}

	/**
	 * Assign ValidationRequest object.
	 * Preferred validation format is JSON-escaped text. Requires \r\n as line breaks!!!
	 * @throws SAREFPipelineException
	 */
	private void createValidationRequest(List<String> shaclFileList) throws SAREFPipelineException {
		createRuleSetList(shaclFileList);
		if (this.ruleSetList.size() < 1) {
			throw new SAREFPipelineException("ShaclValidationManager", "No SHACL files to process!");
		}
		// process ontology file
		try {
			String fileStr = new String(Files.readAllBytes(Paths.get(this.ontologyToValidate)));
			this.validationRequest = new ValidationRequest();
			this.validationRequest.setContentToValidate(fileStr);
			this.validationRequest.setExternalRules(this.ruleSetList);
		} catch (IOException ex) {
			throw new SAREFPipelineException("ShaclValidationManager", "Unable to read ontology file: " + this.ontologyToValidate);
		}
	}

	private HttpRequest.BodyPublisher ofForm(Map<String, Object> data) {
		StringBuilder body = new StringBuilder();
		for (Object dataKey : data.keySet()) {
			if (body.length() > 0) {
				body.append("&");
			}
			body.append(encodeBody(dataKey))
					.append("=")
					.append(encodeBody(data.get(dataKey)));
		}
		return HttpRequest.BodyPublishers.ofString(body.toString());
	}

	// Use 500 Internal server error for failed SHACL validation => value is SHACL file name OR error message.
	private HashMap<String, String> parseHttpResponse(HttpResponse<String> response) {
		String result = response.body().substring(response.body().indexOf("sh:conforms")+13);
		result = result.substring(0,4).replaceAll("\\s+","");
		HashMap<String, String> map = new HashMap<>();
		map.put(String.valueOf(response.statusCode()), result);
		return map;
	}

	/**
	 * POST JSON successful response format:
	 LIST OF: @prefix :        <https://saref.etsi.org/saref4auto/> .
	 [ rdf:type     sh:ValidationReport ;
	 	sh:conforms  true
	 ] .
	 * Generic validator option: 1. Replace {contentToValidate} value with the ontology to validate encoded in JSON. This can also be done with URLs.
	 * 2. Configure the user-provided SHACL shape as a RuleSet element of the externalRules array.
	 * [ "ruleSet": "JSON-encoded_SHACL_shape", "embeddingMethod":"STRING", "ruleSyntax": TEXT_TURTLE ]
	 * POST request to Validate a single RDF ontology.
	 * Return <HTTP status code, validation result>.
	 * @throws SAREFPipelineException
	 * @throws IOException
	 * @throws InterruptedException
	 */
	public HashMap<String, String> ValidateOntologyWithShacl(List<String> shaclFileList) throws SAREFPipelineException, InterruptedException {
		try {
			createValidationRequest(shaclFileList);

			String jsonStr = this.objectMapper.writeValueAsString(this.validationRequest);
			// Had to format the json manually because escapeJson() also escaped double-quotes. See validation_request_using_strings.txt
			//String escapedJson = StringEscapeUtils.escapeJson(jsonStr);
			jsonStr = jsonStr.replace("text/turtle", "text~turtle").replace("application/json", "application~json");
			jsonStr = jsonStr.replace("/","\\/").replace("\\n","\\r\\n");
			jsonStr = jsonStr.replace("text~turtle", "text/turtle").replace("application~json", "application/json");

			HttpRequest request = HttpRequest.newBuilder()
					.header("Content-Type", JSON_FORMAT)
					.uri(URI.create(VALIDATOR_URL + "any/api/validate"))
					.POST(HttpRequest.BodyPublishers.ofString(jsonStr))
					.build();

			HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());
			return parseHttpResponse(response);
		} catch (IOException ex) {
			throw new SAREFPipelineException("ValidateOntologyWithShacl", "Unable to validate ontology: " + this.ontologyToValidate);
		}
	}

}
