/*
 * Copyright 2020 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.checkers;

import java.util.Set;
import java.util.stream.Collectors;

import org.semanticweb.owl.explanation.api.Explanation;
import org.semanticweb.owl.explanation.api.ExplanationGenerator;
import org.semanticweb.owlapi.model.OWLAxiom;
import org.semanticweb.owlapi.model.OWLOntology;
import org.semanticweb.owlapi.model.parameters.Imports;
import org.semanticweb.owlapi.profiles.OWL2DLProfile;
import org.semanticweb.owlapi.profiles.OWLProfileReport;
import org.semanticweb.owlapi.profiles.violations.UseOfDefinedDatatypeInDatatypeRestriction;
import org.semanticweb.owlapi.profiles.violations.UseOfDefinedDatatypeInLiteral;
import org.semanticweb.owlapi.profiles.violations.UseOfUndeclaredDatatype;
import org.semanticweb.owlapi.profiles.violations.UseOfUnknownDatatype;

import fr.mines_stetienne.ci.saref.SAREFPipelineException;
import fr.mines_stetienne.ci.saref.entities.SAREFExample;
import fr.mines_stetienne.ci.saref.managers.OntologyManager;
import fr.mines_stetienne.ci.saref.managers.RepositoryManager;

/**
 * Checks TS 103 673 Clause 9.6.4: OWL Profile, Consistency, and Satisfiability of Classes
 * 
 */
public class Clause_9_6_4_Checker extends AbstractClauseChecker {

	private static enum MESSAGE {
		profile, consistent, satisfiable, exception;
	}

	private final SAREFExample example;

	public Clause_9_6_4_Checker(RepositoryManager repositoryManager, SAREFExample example) {
		super(repositoryManager, Clause_9_6_4_Checker.class, "example " + example.getName());
		this.example = example;
	}

	@Override
	public void checkClause() throws SAREFPipelineException {

		final OWLOntology ontology = pipeline.getOntologyManager().loadOntology(example, errorLogger);
		if (ontology == null) {
			return;
		}

		final OWLProfileReport report = new OWL2DLProfile().checkOntology(ontology);
		report.getViolations().removeIf(v->{
			if(v instanceof UseOfDefinedDatatypeInDatatypeRestriction
					|| v instanceof UseOfDefinedDatatypeInLiteral
					|| v instanceof UseOfUndeclaredDatatype
					|| v instanceof UseOfUnknownDatatype) {
				return true;
			}
			return false;
		});
		
		if(!report.getViolations().isEmpty()) {
			String violations = report.getViolations().stream().map(Object::toString)
					.collect(Collectors.joining("\n- ", "\n\n- ", "\n\n"));
			logError(getMessage(MESSAGE.profile, violations));
		}
		
		OntologyManager ontologyManager = pipeline.getOntologyManager();

		try {
			Set<Explanation<OWLAxiom>> incExplanation = ontologyManager.getInconsistenceExplanations(ontology, errorLogger);
			if (incExplanation != null && !incExplanation.isEmpty()) {
				String explanation = incExplanation.stream().map(e -> e.getAxioms().toString())
						.collect(Collectors.joining("\n- ", "\n\n- ", "\n\n"));
				logError(getMessage(MESSAGE.consistent, explanation));
			} else {
				ExplanationGenerator<OWLAxiom> generator = ontologyManager.getExplanationGenerator(ontology);
				ontology.classesInSignature(Imports.INCLUDED).forEach(c -> {
					if(c.isOWLNothing()) {
						return;
					}
					Set<Explanation<OWLAxiom>> incExplanation2 = ontologyManager.getIncoherenceExplanations(generator, c);
					if (!incExplanation2.isEmpty()) {
						String explanation = incExplanation2.stream().map(e -> e.getAxioms().toString())
								.collect(Collectors.joining("\n- ", "\n\n- ", "\n\n"));
						logError(getMessage(MESSAGE.satisfiable, c, explanation));
		
					}
				});
			}
		} catch (Exception ex) {
			logError(getMessage(MESSAGE.exception), ex);
		}
			
	}

}
