/*
 * Copyright 2020 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref;

import java.util.Arrays;
import java.util.ResourceBundle;

import org.slf4j.Logger;

import fr.mines_stetienne.ci.saref.managers.DatasetManager;
import fr.mines_stetienne.ci.saref.managers.OntologyManager;
import fr.mines_stetienne.ci.saref.managers.SiteManager;
import fr.mines_stetienne.ci.saref.managers.SourcesManager;

public abstract class SAREFErrorLogger {


	protected final SAREFPipeline pipeline;
	protected final DatasetManager datasetManager;
	protected final SourcesManager sourcesManager;
	protected final OntologyManager ontologyManager;
	protected final SiteManager siteManager;
	protected final ResourceBundle bundle;
	
	protected final Logger errorLogger;

	public SAREFErrorLogger(SAREFPipeline pipeline, Logger errorLogger) {
		this.pipeline = pipeline;
		this.datasetManager = pipeline.getDatasetManager();
		this.sourcesManager = pipeline.getSourcesManager();
		this.ontologyManager = pipeline.getOntologyManager();
		this.siteManager = pipeline.getSiteManager();
		this.errorLogger = errorLogger;
		this.bundle = ResourceBundle.getBundle("messages/" + getClass().getSimpleName());
	}
	
	public SAREFPipeline getPipeline() {
		return pipeline;
	}

	protected <T> String getMessage(T object, Object... args) {
		String key = object.toString();
		String msg = bundle.getString(key);
		if (args != null) {
			msg = String.format(msg, args);
		}
		return msg;
	}

	protected void log(String msg, SAREFPipeline.Mode... errorModes) {
		log(errorLogger, msg, errorModes);
	}

	protected <T> void log(String msg, Throwable t, SAREFPipeline.Mode... errorModes) {
		log(errorLogger, msg, t, errorModes);
	}

	protected void logWarning(String msg) {
		errorLogger.warn(msg);
	}
	
	protected void logWarning(String msg, Throwable t) {
		errorLogger.warn(msg, t);
	}
	
	protected void logError(String msg) {
		errorLogger.error(msg);
	}
	
	protected void logError(String msg, Throwable t) {
		errorLogger.error(msg, t);
	}
	
	protected void log(Logger logger, String msg, SAREFPipeline.Mode... errorModes) {
		if (Arrays.asList(errorModes).contains(pipeline.mode)) {
			logger.error(msg);
		} else {
			logger.warn(msg);
		}
	}
	
	protected <T> void log(Logger logger, String msg, Throwable t, SAREFPipeline.Mode... errorModes) {
		if (Arrays.asList(errorModes).contains(pipeline.mode)) {
			logger.error(msg, t);
		} else {
			logger.warn(msg, t);
		}
	}
}
