/*
 * Copyright 2020 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.tests;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlType;

/**
 * @author Omar Qawasmeh
 */
@XmlRootElement(name = "testsuites")
@XmlType(propOrder = { "name", "tests", "failures", "errors", "testSuitesList" })
public class TestSuites implements Serializable {

	@XmlAttribute
	String name = "";

	@XmlAttribute
	int tests = 0;

	@XmlAttribute
	int failures = 0;
	
	@XmlAttribute
	int errors = 0;
	
	@XmlElement(name="testsuite")
	private List<TestSuite> testSuitesList = new ArrayList<>();

	private transient Map<String, TestSuite> testSuitesMap;
	
	public TestSuites() {}
	
	public TestSuites(String name) {
		this.name = name;
	}
	
	public String getName() {
		return name;
	}

	public int getFailures() {
		return failures;
	}

	public int getErrors() {
		return errors;
	}

	public TestSuite get(String name) {
		if(testSuitesMap == null) {
			testSuitesMap = new HashMap<>();
			for(TestSuite testSuite : testSuitesList) {
				testSuitesMap.put(testSuite.getName(), testSuite);
			}
		}
		TestSuite testSuite = testSuitesMap.get(name);
		if(testSuite == null) {
			testSuite = new TestSuite(this, name);
			testSuitesMap.put(name, testSuite);
			testSuitesList.add(testSuite);
		}
		return testSuite;
	}

	public void clean() {
		testSuitesList.removeIf((ts) -> {
			return ts.getErrors() + ts.getFailures() == 0;
		});		
		if(testSuitesMap != null && testSuitesMap.values() != null) {
			testSuitesMap.values().removeIf((ts) -> {
				return ts.getErrors() + ts.getFailures() == 0;
			});
		}
		
		testSuitesList.forEach((ts)->{
			// put errors first
			Collections.sort(ts.getTestCases(), new Comparator<TestCase>() {
				@Override
				public int compare(TestCase o1, TestCase o2) {
					if (o1.getStatus().equals(o2.getStatus())) {
						return 0;
					}
					if (o1.getStatus().equals(TestCase.Status.ERROR.getName())) {
						return -1;
					}
					if (o2.getStatus().equals(TestCase.Status.ERROR.getName())) {
						return 1;
					}
					return 0;
				}
			});
			
		});
	}

}