/*
 * Copyright 2020 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.entities;

import org.apache.jena.rdf.model.Resource;

import fr.mines_stetienne.ci.saref.utils.Languages;

/**
 *
 * @author maxime.lefrancois
 */
public interface SAREFProject {

	/**
	 * Gets the SAREF project name.
	 * <ul>
	 * <li>The name of SAREF core is SAREF;</li>
	 * <li>the name of a SAREF extension is SAREF4ABCD, where ABCD is the SAREF
	 * extension acronym.</li>
	 * </ul>
	 * 
	 * @return
	 */
	public String getName();

	/**
	 * Gets the SAREF project repository name:
	 * <ul>
	 * <li>for SAREF core, it is saref-core</li>
	 * <li>for SAREF4ABCD, it is saref4abcd</li>
	 * </ul>
	 * 
	 * @return
	 */
	public String getRepositoryName();

	/**
	 * Gets the SAREF project prefix:
	 * <ul>
	 * <li>for SAREF core, it is saref</li>
	 * <li>for SAREF4ABCD, it is saref4abcd</li>
	 * </ul>
	 * 
	 * @return
	 */
	public String getPrefix();

	/**
	 * Gets the SAREF project path:
	 * <ul>
	 * <li>for SAREF core, it is core</li>
	 * <li>for SAREF4ABCD, it is saref4abcd</li>
	 * </ul>
	 * 
	 * @return
	 */
	public String getPath();

	/**
	 * Gets the SAREF project namespace:
	 * <ul>
	 * <li>for SAREF core, it is https://saref.etsi.org/core/</li>
	 * <li>for SAREF4ABCD, it is https://saref.etsi.org/saref4abcd/</li>
	 * </ul>
	 * 
	 * @return
	 */
	public String getNamespace();

	/**
	 * Gets the turtle file name:
	 * <ul>
	 * <li>for SAREF core, it is saref</li>
	 * <li>for SAREF4ABCD, it is saref4abcd</li>
	 * </ul>
	 * 
	 * @return
	 */
	public String getOntologyFileName();

	/**
	 * Gets the ontology file name for a certain language:
	 * <ul>
	 * <li>for SAREF core and RDF/XML, it is saref.rdf</li>
	 * <li>for SAREF4ABCD and RDF/XML, it is saref4abcd.rdf</li>
	 * </ul>
	 * 
	 * @return
	 */
	public String getOntologyFileName(Languages l);
	
	/**
	 * Gets the Jena resource:
	 * <ul>
	 * <li>for SAREF core, it is https://saref.etsi.org/core/</li>
	 * <li>for SAREF4ABCD, it is https://saref.etsi.org/saref4abcd/</li>
	 * </ul>
	 * 
	 * @return
	 */
	public Resource getResource();

}