/*
 * Copyright 2024 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.checkers;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import fr.mines_stetienne.ci.saref.SAREFPipelineException;
import fr.mines_stetienne.ci.saref.managers.RepositoryManager;

/**
 * Checks TS 103 673 Clause 9.7.1: documentation files.
 * 
 */
public class Clause_9_7_1_Checker extends AbstractClauseChecker {

	private enum MESSAGE {
		ioexception
	}

	private static final Map<String, String> DEFAULTVALUES = new HashMap<String, String>();
	static {
		DEFAULTVALUES.put("creators.html", "- [Maxime Lefrançois](http://maxime-lefrancois.info/me#) ([MINES Saint-Étienne](https://www.mines-stetienne.fr/))");
		DEFAULTVALUES.put("contributors.html", "<!--TS 103 673 v0.3.4 Clause 9.7.1-->\n" +
				"<!--The optional documentation source contributors.html or contributors.md shall describe the list of any contributor. The Steering Board and project leader are responsible for approving the list of contributors. The set of contributors in the ontology document shall correspond to the list of contributors in this document source.-->\n" +
				"<ul>\n" +
				"  <li><a href=\"http://www.maxime-lefrancois.info/\">Maxime Lefrançois</a> (<a href=\"http://www.mines-telecom.fr/en/\">Institut Mines-Télécom - MINES Saint-Étienne</a>)</li>\n" +
				"</ul>");
		DEFAULTVALUES.put("abstract.html", "<!--TS 103 673 v0.3.4 Clause 9.7.1-->\n" +
				"<!--The optional documentation source abstract.html or abstract.md shall contain a short description of the SAREF project version ontology. It shall not include diagrams.-->\n" +
				"<p>The objective of SAREF4abcd is to extend <a href=\"https://saref.etsi.org/core/\">SAREF ontology</a> for the abcd domain. </p>");
		DEFAULTVALUES.put("description.html", "<h3>General Overview</h3>\n" +
				"\n" +
				"<p>The documentation for Unknown is missing. Please refer to the corresponding technical specification for more information.</p> \n" +
				"\n" +
				"<strong>ETSI Technical Specification:</strong> <a href=\"https://www.etsi.org/deliver/etsi_ts/unknown/unknown.pdf\">ETSI TS <unknown> : \"SmartM2M; Extension to SAREF; Part 7: unknown Domain\"</a>");
		DEFAULTVALUES.put("examples.html", "<!--TS 103 673 v0.3.4 Clause 9.7.1-->\n" +
				"<!--The optional documentation source examples.html or examples.md shall contain complete examples. It should include diagrams as defined in clause 9.6.2.-->");
		DEFAULTVALUES.put("references.html", "<h3>Normative references</h3>\n" +
				"<ul>\n" +
				"  <li id=\"[1]\">[1] ETSI TS 103 264 (V3.1.1) (02-2020): \"SmartM2M; Smart Applications; Reference Ontology and oneM2M Mapping\".</li>\n" +
				"</ul>\n" +
				"<h3>Informative references</h3>\n" +
				"<ul>\n" +
				"</ul>");
		DEFAULTVALUES.put("acknowledgement.html", "<!--TS 103 673 v0.3.4 Clause 9.7.1-->\n" +
				"<!--The optional documentation source acknowledgement.html or acknowledgement.md shall contain a list of acknowledgements. It shall not include diagrams.-->");
	}

	public Clause_9_7_1_Checker(RepositoryManager repositoryManager) {
		super(repositoryManager, Clause_9_7_1_Checker.class);
	}

	@Override
	public void checkClause() throws SAREFPipelineException {
		File dir = new File(repository.getDirectory(), "documentation");
		if (!dir.isDirectory()) {
			try {	// create all default files
				final Map<String, String> defaultValues = version.getModel().getNsPrefixMap();
				for (String s : DEFAULTVALUES.keySet()) {
					String[] lines = {defaultValues.get(s)};
					CreateFileInDirectoryWithContents("documentation", DEFAULTVALUES.get(s), lines);
				}
			} catch (IOException ex) {
				logError(getMessage(Clause_9_7_1_Checker.MESSAGE.ioexception));
			}
		}
// These are optional files so do NOT verify the html text within the file!!!
	}
}
