/*
 * Copyright 2020 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.checkers;

import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import fr.mines_stetienne.ci.saref.SAREF;
import fr.mines_stetienne.ci.saref.SAREFPipelineException;
import fr.mines_stetienne.ci.saref.entities.SAREFExtension;
import fr.mines_stetienne.ci.saref.managers.RepositoryManager;

/**
 * Checks TS 103 673 Clause 9.4.2: Prefixes declaration
 * 
 */
public class Clause_9_4_2_Checker extends AbstractClauseChecker {

	private final static Pattern PATTERN_PREFIX = Pattern.compile(SAREF.REGEX_EXTENSION_PREFIX);

	private static final Map<String, String> PREFIXES = new HashMap<String, String>();
	static {
		PREFIXES.put("rdf", "http://www.w3.org/1999/02/22-rdf-syntax-ns#");
		PREFIXES.put("rdfs", "http://www.w3.org/2000/01/rdf-schema#");
		PREFIXES.put("owl", "http://www.w3.org/2002/07/owl#");
		PREFIXES.put("xsd", "http://www.w3.org/2001/XMLSchema#");
		PREFIXES.put("dcterms", "http://purl.org/dc/terms/");
		PREFIXES.put("vann", "http://purl.org/vocab/vann/");
		PREFIXES.put("voaf", "http://purl.org/vocommons/voaf#");
		PREFIXES.put("schema", "http://schema.org/");
		PREFIXES.put("geo", "http://www.opengis.net/ont/geosparql#");
		PREFIXES.put("saref", "https://saref.etsi.org/core/");
	}

	private static enum MESSAGE {
		different, expect;
	}

	public Clause_9_4_2_Checker(RepositoryManager repositoryManager) {
		super(repositoryManager, Clause_9_4_2_Checker.class);
	}

	@Override
	public void checkClause() throws SAREFPipelineException {
		final Map<String, String> prefixes = version.getModel().getNsPrefixMap();
		for (String s : PREFIXES.keySet()) {
			if (prefixes.containsKey(s)) {
				if (!prefixes.get(s).equals(PREFIXES.get(s))) {
					logError(getMessage(MESSAGE.different, s, PREFIXES.get(s), prefixes.get(s)));
				}
			}
		}
		boolean found = false;
		for (String s : prefixes.keySet()) {
			Matcher m = PATTERN_PREFIX.matcher(s);
			if (m.matches()) {
				String acronym = m.group(SAREF.REGEX_ACRONYM_VAR);
				String expected = new SAREFExtension(acronym).getNamespace();
				if(!prefixes.get(s).equals(expected)) {
					logError(getMessage(MESSAGE.different, s, expected, prefixes.get(s)));
				}
			}
			if (s.equals(repository.getPrefix())) {
				found = true;
			}
		}
		if (!found) {
			String namespace = repository.getNamespace();
			String prefix = repository.getPrefix();
			logError(getMessage(MESSAGE.expect, namespace, prefix));
		}
		// how to check MESSAGE.BASE ?
	}
}
