/*
 * Copyright 2024 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.checkers;

import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.jena.rdf.model.Model;
import org.apache.jena.rdf.model.Resource;
import org.apache.jena.vocabulary.DCTerms;
import org.apache.jena.vocabulary.RDF;

import com.opencsv.CSVParser;
import com.opencsv.CSVParserBuilder;
import com.opencsv.CSVReader;
import com.opencsv.CSVReaderBuilder;

import fr.mines_stetienne.ci.saref.SAREFPipelineException;
import fr.mines_stetienne.ci.saref.managers.GenerateRDFaManager;
import fr.mines_stetienne.ci.saref.managers.RepositoryManager;
import fr.mines_stetienne.ci.saref.utils.Languages;
import fr.mines_stetienne.ci.saref.vocabs.VTC;

/**
 * Checks TS 103 673 Clause 9.3: Ontology requirements
 * 
 */
public class Clause_9_3_Checker extends AbstractClauseChecker {

	private static final String FIRST_LINE = "Id;Category;Requirement";

	private static enum MESSAGE {
		missing, ioexception, line;
	}

	public Clause_9_3_Checker(RepositoryManager repositoryManager) {
		super(repositoryManager, Clause_9_3_Checker.class);
	}

	@Override
	public void checkClause() throws SAREFPipelineException {
		File dir = new File(repository.getDirectory(), "requirements");
		if (!dir.isDirectory()) {
			//return;
			try {
				String[] lines = {FIRST_LINE};
				CreateFileInDirectoryWithContents("requirements", "requirements.csv", lines);
			} catch (IOException ex) {
				logError(getMessage(Clause_9_3_Checker.MESSAGE.ioexception));
			}
		}
		try {
			File file = new File(dir, "requirements.csv");
			if (!file.exists()) {
				logError(getMessage(MESSAGE.missing));
				return;
			}
			List<String> lines = FileUtils.readLines(file, StandardCharsets.UTF_8);
			if (lines.size() < 1 || !lines.get(0).equals(FIRST_LINE)) {
				logError(getMessage(MESSAGE.line));
				return;
			}
			readRequirements();
			requirementsRDFaGenerator();
		} catch (IOException ex) {
			logError(getMessage(MESSAGE.ioexception), ex);
		}
	}

	public void readRequirements() throws SAREFPipelineException {
		Model requirements = version.getRequirements();
		File file = new File(repository.getDirectory(), "requirements/requirements.csv");
		CSVParser parser = new CSVParserBuilder().withSeparator(';').withQuoteChar('"').build();
		try (FileReader filereader = new FileReader(file);
				CSVReader csvReader = new CSVReaderBuilder(filereader).withCSVParser(parser).build()) {
			csvReader.forEach(row -> {
				String id = row[0]; // WATR-1
				Resource resource = requirements.getResource(String.format("%srequirements#%s", version.getIRI(), id));
				String category = row[1]; // Water infrastructure
				String requirement = row[2]; // Which assets compose a water distribution infrastructure? ...
				requirements.add(resource, RDF.type, VTC.Requirement);
				requirements.add(resource, VTC.requirementId, id);
				requirements.add(resource, VTC.category, category);
				requirements.add(resource, DCTerms.description, requirement);
			});
		} catch (Exception e) {
			logWarning(getMessage(MESSAGE.ioexception), e);
		}
	}

	private void requirementsRDFaGenerator() throws SAREFPipelineException, IOException {
		if (pipeline.ignoreSite) {
			return;
		}
		String categoryChanger = "";
		String repoName = project.getName();
		String href = project.getNamespace();
		File reqCSV = new File(repository.getDirectory(), "/requirements/requirements.csv");
		File versionSite = new File(siteManager.siteDir, version.getVersionPath());
		FileUtils.forceMkdir(versionSite);
		File reqHTML = new File(versionSite, "requirements.html");
		GenerateRDFaManager.GenerateRDFa(categoryChanger, repoName, href, reqCSV, reqHTML, "requirements");
		Model model = version.getRequirements();
		for (Languages l : Languages.values()) {
			String fileName = String.format("requirements.%s", l.getExt());
			File file = new File(versionSite, fileName);
			try (FileOutputStream fos = new FileOutputStream(file)) {
				model.write(fos, l.getLang());
			} catch (IOException ex) {
				String msg = getMessage(MESSAGE.ioexception, ex);
				errorLogger.warn(msg, ex);
			}
		}
	}

}
