/*
 * Copyright 2024 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.managers;

import java.io.IOException;
import java.net.URI;
import java.net.URLEncoder;
import java.net.http.*;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.fasterxml.jackson.databind.ObjectMapper;
//import fr.mines_stetienne.ci.saref.SAREFErrorLogger;

public class ShaclValidationManager {

	private final static String BASE_URL = "http://localhost:8080/shacl/";  // <<<
	private String ontologyToValidate;
	private final HttpClient client;
	private final ObjectMapper objectMapper;

	private static String encode(Object obj) {
		return URLEncoder.encode(obj.toString(), StandardCharsets.UTF_8);
	}

	private enum MESSAGE {
		http_exception, ioexception
	}

	// The full list of available tags: https://www.itb.ec.europa.eu/docs/guides/latest/validatingRDF/
	// Replace default values for {contentToValidate, validationType}
	private String shaclRequestBody = "{ "+	// generic validator option
		"'contentToValidate': 'string', "+	// entire content of ontology file (or URI)
		"'embeddingMethod': 'STRING', "+  	// {STRING,URL,BASE64}
		"'contentSyntax': 'text/turtle', "+
		"'validationType': 'string', "+	// 	Clause_9_4_3_1-3.ttl, Clause_9_4_4_2.ttl, Clause_9_6_3.ttl (for configured validator)
		"'reportSyntax': 'text/turtle', "+
		"'locale': 'en', "+
		"'rdfReportSyntax': 'application/json' "+
		"}";
	JsonObject jsonObject = new JsonParser().parse(shaclRequestBody).getAsJsonObject();

	/**
	 * Open a REST interface to the SHACL validator.
	 */
	public ShaclValidationManager(String ontologyToValidate) throws IOException {
		client = HttpClient.newHttpClient();
		objectMapper = new ObjectMapper();
		this.ontologyToValidate = ontologyToValidate;	//<<< test for file presence.
	}

	public static HttpRequest.BodyPublisher ofForm(Map<String, String> data) {
		StringBuilder body = new StringBuilder();
		for (String dataKey : data.keySet()) {
			if (body.length() > 0) {
				body.append("&");
			}
			body.append(encode(dataKey))
					.append("=")
					.append(encode(data.get(dataKey)));
		}
		return HttpRequest.BodyPublishers.ofString(body.toString());
	}

	/**
	 * POST request to Validate a single RDF instance: {domain}/api/validate/{requestBody}
	 <<< This method uses the validator configured to read the Clause_9_4_3_1-3.ttl, Clause_9_4_4_2.ttl, Clause_9_6_3.ttl shacl files.
	 * Response: HTTP code 200 for successful validation as JSON.
	 */
	public HttpResponse<String> validateOntologyWithShacl()
			throws IOException, InterruptedException {
		Map<String, String> data = new HashMap<>();
		data = objectMapper.readValue(shaclRequestBody, HashMap.class);
		data.put("contentToValidate", ontologyToValidate);	// file:///home/davidgnabasik/dev/real_saref4auto/ontology/saref4auto.ttl or URL
		data.put("validationType", validationType);
		System.out.println("Map is: "+data);//<<<
				/* "contentToValidate": "RDF_CONTENT_AS_BASE64_ENCODED_STRING",
				"contentSyntax": "text/turtle",
				"embeddingMethod": "STRING",
				"validationType": "Clause_9_4_3_1",
				"reportSyntax": "application/json",
				"locale": "en" */

		HttpRequest request = HttpRequest.newBuilder()
				.header("Content-Type", "text/turtle")	// application/x-www-form-urlencoded
				.uri(URI.create(BASE_URL + validationType + "/api/validate" ))
				.POST(ofForm(data))
				.build();

		HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());
		System.out.println("Status code: " + response.statusCode());
		System.out.println("\n Body: " + response.body());

		return response;
	}

	/**
	 * If shaclDomain is empty, return info for all defined domains.
	 * @throws IOException
	 * @throws InterruptedException
	 */
	public HttpResponse<String> getShaclDomainInfo(String shaclDomain) throws IOException, InterruptedException {
		Map<String, String> data = new HashMap<>();
		data = objectMapper.readValue(shaclRequestBody, HashMap.class);
		data.put("validationType", shaclDomain);	// {Ontology,Commodity}}

		String url = BASE_URL + "shacl/api/info";
		if (shaclDomain.length() > 0)	 url = BASE_URL + "shacl/" + shaclDomain + "/api/info";

		HttpRequest request = HttpRequest.newBuilder()
				.header("Accept", "text/turtle")
				.header("Content-Type", "text/turtle")
				.uri(URI.create(url))
				.GET()
				.build();

		HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());
		System.out.println("Status code: " + response.statusCode());
		System.out.println("\n Body: " + response.body());

		return response;
	}
}
/*
[
		{
		"domain": "Commodity",
		"validationTypes": [
		{
		"type": "Commodity",
		"description": "Commodity"
		}
		]
		},
		{
		"domain": "Ontology",
		"validationTypes": [
		{
		"type": "Clause_9_4_3_1",
		"description": "Clause_9_4_3_1"
		},
		{
		"type": "Clause_9_4_3_2",
		"description": "Clause_9_4_3_2"
		},
		{
		"type": "Clause_9_4_3_3",
		"description": "Clause_9_4_3_3"
		},
		{
		"type": "Clause_9_4_3_4",
		"description": "Clause_9_4_3_4"
		},
		{
		"type": "Clause_9_6_3",
		"description": "Clause_9_6_3"
		},
		{
		"type": "ontologyPerson",
		"description": "ontologyPerson"
		}
		]
		}
		]
		*/
/* POST JSON report format:
{
  "date": "2024-04-05T10:18:24.197+0000",
  "result": "SUCCESS",
  "overview": {
    "profileID": "Clause_9_4_3_1"
  },
  "counters": {
    "nrOfAssertions": 0,
    "nrOfErrors": 0,
    "nrOfWarnings": 0
  },
  "context": {},
  "reports": {},
  "name": "SHACL Validation"
}
 */
