/*
 * Copyright 2024 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.checkers;

import java.io.*;
import java.nio.file.Files;
import java.util.Set;
import java.util.ArrayList;
import java.util.Iterator;

import org.semanticweb.owlapi.model.OWLImportsDeclaration;
import org.semanticweb.owlapi.model.OWLOntology;
import org.semanticweb.owlapi.model.IRI;

import fr.mines_stetienne.ci.saref.entities.SAREFExtension;
import fr.mines_stetienne.ci.saref.SAREFPipelineException;
import fr.mines_stetienne.ci.saref.managers.RepositoryManager;
import fr.mines_stetienne.ci.saref.utils.Languages;

/**
 * Checks TS 103 673 Clause 9.8.1: Vocabularies ontology directory. Need at least one file.
 * 
 */
public class Clause_9_8_1_Checker extends AbstractClauseChecker {

	private Set<OWLImportsDeclaration> importDeclarations;

	private static enum MESSAGE {
		error, missing, ioexception, line, one, name, turtle;
	}

	public Clause_9_8_1_Checker(RepositoryManager repositoryManager, Set<OWLImportsDeclaration> importDeclarations) {
		super(repositoryManager, Clause_9_8_1_Checker.class);
		this.importDeclarations = importDeclarations;
	}

	@Override
	public void checkClause() throws SAREFPipelineException {
		final String directoryName = "vocabularies";
		File dir = new File(repository.getDirectory(), directoryName);
		if (!dir.isDirectory()) {
			return;
		}

		// Step 4. Fetch the list of turtle files in the vocabularies directory.
		File[] fileList = null;
		try {
			if (Files.walk(dir.toPath(), 1).filter(p -> !p.toFile().isFile() && !p.toFile().getName().startsWith("."))
					.count() != 1) {
				logError(getMessage(Clause_9_8_1_Checker.MESSAGE.one, repository.getProject().getOntologyFileName(Languages.TEXT_TURTLE)));
			}
			// Collect the available vocabulary files.
			FileFilter filter = new FileFilter() {
				public boolean accept(File f)
				{
					return f.getName().endsWith("ttl");
				}
			};
			fileList = dir.listFiles(filter);
		} catch (IOException e) {
			logError(getMessage(Clause_9_8_1_Checker.MESSAGE.ioexception), e);
		}
		// Step 5. If the list is empty, return an error.
		if (fileList.length < 1) {
			log(getMessage(Clause_9_8_1_Checker.MESSAGE.missing));
			return;
		}

		// <<< Step 6. Match each imports statements to the presence of each file using the last tag (e.g. "properties" +".ttl")
		//		Ignore turtle files which do not have a corresponding imports statements.
		/*Iterator<OWLImportsDeclaration> itr = this.importDeclarations.iterator();
		while(itr.hasNext()) { // <https://saref.etsi.org/saref4auto/v2.1.1/properties/>
			IRI iri = itr.next().getIRI();
		}*/

		// Step 7. Read each vocabulary file and check its syntax.
		for(int i = 0; i < fileList.length; i++) {
			String fileName = "file://" + fileList[i].getAbsolutePath();
			// SAREFExtension(String acronym, name, path, prefix, namespace, ontologyFileName)	use dummy values
			SAREFExtension vocabulary = new SAREFExtension("abcd", "abcd", "owl", "owl", "abcd", fileName);
			IRI iri = IRI.create(fileName);
			// Ignore ==> ERROR in org.semanticweb.owlapi.rdf.rdfxml.parser.OWLRDFConsumer: Entity not properly recognized...
			final OWLOntology ontology = pipeline.getOntologyManager().loadOntology(vocabulary, errorLogger, iri);
			if (ontology == null) {
				return;
			}
			try {
				new Clause_9_8_1_1_Checker(repositoryManager).check();
				new Clause_9_8_1_2_Checker(repositoryManager).check();
				new Clause_9_8_1_3_Checker(repositoryManager).check();
				new Clause_9_8_1_4_Checker(repositoryManager).check();
				new Clause_9_8_1_5_Checker(repositoryManager).check();
			} catch (SAREFPipelineException ex) {
				logError(getMessage(Clause_9_8_1_Checker.MESSAGE.error), ex);
			}
		} // for loop
	}

}
