/*
 * Copyright 2024 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.checkers;

import java.io.IOException;
import java.net.http.HttpResponse;

import org.apache.jena.rdf.model.Model;
import org.apache.jena.rdf.model.ResIterator;
import org.apache.jena.rdf.model.Resource;
import org.apache.jena.vocabulary.DCTypes;
import org.apache.jena.vocabulary.RDF;

import fr.mines_stetienne.ci.saref.SAREF;
import fr.mines_stetienne.ci.saref.SAREFPipelineException;
import fr.mines_stetienne.ci.saref.entities.SAREFExample;
//import fr.mines_stetienne.ci.saref.entities.SAREFVersion;
import fr.mines_stetienne.ci.saref.entities.SAREFVersionName;
import fr.mines_stetienne.ci.saref.managers.RepositoryManager;
import fr.mines_stetienne.ci.saref.vocabs.SHACL;
import fr.mines_stetienne.ci.saref.managers.ShaclValidationManager;

/**
 * Checks TS 103 673 Clause 9.4.6: Conformance to reference ontology patterns.
 * The ontology in the ontology document of a SAREF project version shall conform to the SHACL specification of each
 * reference ontology pattern defined in this SAREF project version.
 */
public class Clause_9_4_6_Checker extends AbstractShaclChecker {

	private enum MESSAGE implements MessageResource {
		dataset, conformsTo1, conformsToNot, conformsToNot2, conformsTo2, conformsTo3,
		title1, title2, title3, abstract1, description1, description2,
		license, ioexception
	}

	private final SAREFExample example = null;

	public Clause_9_4_6_Checker(RepositoryManager repositoryManager) //, SAREFExample shacl
			throws SAREFPipelineException {
		super(repositoryManager, Clause_9_4_6_Checker.class);	// , "example " + example.getName());
		//this.example = shacl;
	}

	@Override
	protected Model getModel() {
		Model model = example.getModel();
		final String defaultVersion = "v0.0.1";
		final String regex = "^" + example.getIRI().replace(defaultVersion, SAREF.REGEX_VERSION_NUMBER) + "$";
		int onto = 0;
		boolean found = false;
		for (ResIterator it = model.listSubjectsWithProperty(RDF.type,  DCTypes.Dataset); it.hasNext();) {
			Resource r = it.next();
			if(versionName.equals(SAREFVersionName.DEFAULT)) {
				if(r.getURI().matches(regex)) {
					found = true;
				}				
			} else {
				if(r.getURI().equals( example.getIRI() )) {
					found = true;
				}				
			}
			onto++;
		}
		if (onto != 1 || !found) {
			final String msg;
			if(versionName.equals(SAREFVersionName.DEFAULT)) {
				msg = getMessage(MESSAGE.dataset, example.getIRI()).replace("v0.0.1", "<<some SAREF version name>>");
			} else {
				msg = getMessage(MESSAGE.dataset, example.getIRI());
			}
			logError(msg);
		}	
		return model;
	}

	protected final void updateShapeModel() {
		if(!versionName.equals(SAREFVersionName.DEFAULT)) {
			shapeModel.add(MESSAGE.conformsToNot.asResource(), SHACL.pattern, exactly(version.getIRI()));
			add(MESSAGE.conformsTo1, version.getIRI());
		}
		add(MESSAGE.conformsTo2);
		add(MESSAGE.conformsTo3);
		add(MESSAGE.title1);
		add(MESSAGE.title2);
		add(MESSAGE.title3);
		add(MESSAGE.abstract1);
		add(MESSAGE.description1);
		add(MESSAGE.description2);
		add(exactly(SAREF.LICENSE), MESSAGE.license);
		
		if(versionName.equals(SAREFVersionName.DEFAULT)) {
			remove(MESSAGE.conformsTo1);
		}
	}

	/**
	 The ontology in the ontology document of a SAREF project version shall conform to the SHACL specification of each
	 reference ontology pattern defined in this SAREF project version.
	 If the ontology in the ontology document of a SAREF project version imports an {Ontology | Vocabulary | Pattern} Version IRI defined in
	 another SAREF project version, either directly or transitively, then it shall conform to the SHACL specification of each
	 reference ontology pattern defined in this other SAREF project version.
	 This method calls the configured docker image that runs the common shacl files.
	 * */
	protected final void validateOntologyWithShacl(String ontologyToValidate) {
		try {
			HttpResponse<String> response = new ShaclValidationManager(ontologyToValidate).validateOntologyWithShacl();
		} catch (IOException | InterruptedException e) {
			logError(getMessage(Clause_9_4_6_Checker.MESSAGE.ioexception));
		}
	}

	// This method calls the generic docker image that runs a specific shacl files.
	protected final void validateOntologyPerShacl(String ontologyToValidate, String shaclFile) {
		try {
			HttpResponse<String> response = new ShaclValidationManager(ontologyToValidate).validateOntologyWithShacl();
		} catch (IOException | InterruptedException e) {
			logError(getMessage(Clause_9_4_6_Checker.MESSAGE.ioexception));
		}
	}

}