/*
 * Copyright 2024 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.checkers;

import java.io.*;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;

import fr.mines_stetienne.ci.saref.SAREF;
import fr.mines_stetienne.ci.saref.SAREFPipelineException;
import fr.mines_stetienne.ci.saref.managers.RepositoryManager;

public abstract class AbstractClauseChecker extends AbstractChecker {

	private static final String CLAUSE = "clause";
	private static final String CLAUSE_FOR = "clause_for";
	private static final Pattern PATTERN = Pattern.compile("^Clause(?<clause>(_[0-9]+)+)_Checker$");

	private final static Logger getLogger(RepositoryManager repositoryManager, Class<?> clazz, String file) {
		final Matcher matcher = PATTERN.matcher(clazz.getSimpleName());
		final String clause;
		if (!matcher.find()) {
			clause = "unknown";
		} else {
			clause = matcher.group(CLAUSE).substring(1).replace('_', '.');
		}
		final String repositoryName = repositoryManager.getRepository().getName();
		final String versionName;
		if (repositoryManager.getCurrentVersionName() == null) {
			versionName = "SNAPSHOT";
		} else {
			versionName = repositoryManager.getCurrentVersionName().toString();
		}

		final String loggerName;
		if (file == null) {
			loggerName = SAREF.getMessage(CLAUSE, repositoryName, versionName, clause);
		} else {
			loggerName = SAREF.getMessage(CLAUSE_FOR, repositoryName, versionName, file, clause);
		}
		return repositoryManager.getPipeline().getLogger(loggerName);
	}

	public AbstractClauseChecker(RepositoryManager repositoryManager, Class<?> clazz) {
		this(repositoryManager, clazz, null);
	}

	public AbstractClauseChecker(RepositoryManager repositoryManager, Class<?> clazz, String file) {
		super(repositoryManager, getLogger(repositoryManager, clazz, file));
	}

	protected abstract void checkClause() throws SAREFPipelineException;

	public final void check() throws SAREFPipelineException {
		this.checkClause();
	}

	/**
	 * If the directory does not exist, create it. directoryPath can be relative or absolute.
	 * If the file does not exist, create it with the default contents. Creates a file of length=0 if fileContents is empty.
	 * @param directoryPath
	 * @throws IOException
	 */
	public final void CreateDirectory(String directoryPath) throws IOException {
		File theDir = new File(directoryPath);
		if (!theDir.exists()) {
			try {
				theDir.mkdirs();
			} catch (Exception se) {
				throw new IOException("Unable to create directory " + directoryPath);
			}
		}
	}

	/**
	 * If the directory does not exist, create it. directoryPath can be relative or absolute.
	 * If the file does not exist, create it with the default contents. Creates a file of length=0 if fileContents is empty.
	 * @param directoryPath
	 * @param fileName
	 * @param fileContents
	 * @throws IOException
	 */
	public final void CreateFileInDirectoryWithContents(String directoryPath, String fileName, String[] fileContents) throws IOException {
		try {
			CreateDirectory(directoryPath);
		} catch (Exception se) {
			return;
		}
		String fName = directoryPath + "/" + fileName;
		File fout = new File(fName);  // implicit bw.close().
		try (FileOutputStream fos = new FileOutputStream(fout); BufferedWriter bw = new BufferedWriter(new OutputStreamWriter(fos));) {
			for (String str : fileContents) {
				bw.write(str);
				bw.newLine();
			}
		} catch (IOException e) {
			throw new IOException("Unable to create file " + fName);
		}
	}

}