/*
 * Copyright 2020 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.entities;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.jena.rdf.model.Model;
import org.apache.jena.rdf.model.ModelFactory;
import org.apache.jena.vocabulary.DCTerms;
import org.eclipse.jgit.lib.Ref;

import fr.mines_stetienne.ci.saref.utils.Languages;
import fr.mines_stetienne.ci.saref.vocabs.VTC;

public class SAREFVersion extends SAREFNamedGraph implements SAREFProject {

	private final SAREFRepository repository;
	private final SAREFVersionName versionName;
	private final Ref developRef;
	private final Ref prereleaseRef;
	private final Ref releaseRef;
	private final Map<String, SAREFExample> examples = new HashMap<>();
	private final Set<SAREFTerm> definedTerms = new HashSet<>();
	private final Set<SAREFTerm> usedTerms = new HashSet<>();
	private final Model requirements = ModelFactory.createDefaultModel();
	private final Model tests = ModelFactory.createDefaultModel();

	public SAREFVersion(SAREFRepository repository, SAREFVersionName versionName, Ref developRef, Ref prereleaseRef,
			Ref releaseRef) {
		super(String.format("%s%s/", repository.getNamespace(), versionName));
		this.repository = repository;
		this.versionName = versionName;
		this.developRef = developRef;
		this.prereleaseRef = prereleaseRef;
		this.releaseRef = releaseRef;
		
		requirements.setNsPrefix(VTC.PREFIX, VTC.NS);
		requirements.setNsPrefix("dcterms", DCTerms.NS);
		tests.setNsPrefix(VTC.PREFIX, VTC.NS);
	}

	public SAREFRepository getRepository() {
		return repository;
	}

	public SAREFVersionName getVersionName() {
		return versionName;
	}

	public Ref getDevelopRef() {
		return developRef;
	}

	public Ref getPrereleaseRef() {
		return prereleaseRef;
	}

	public Ref getReleaseRef() {
		return releaseRef;
	}

	public Set<SAREFTerm> getDefinedTerms() {
		return definedTerms;
	}

	public Set<SAREFTerm> getUsedTerms() {
		return usedTerms;
	}

	public Map<String, SAREFExample> getExamples() {
		return examples;
	}

	@Override
	public String getName() {
		return repository.getName();
	}

	@Override
	public String getRepositoryName() {
		return repository.getRepositoryName();
	}

	@Override
	public String getPrefix() {
		return repository.getPrefix();
	}

	@Override
	public String getPath() {
		return repository.getPath();
	}

	/**
	 * Gets the version path
	 * <ul>
	 * <li>The version path of SAREF core v1.1.1 is core/v1.1.1;</li>
	 * <li>The version path of SAREF4ABCD v1.1.1 is saref4abcd/v1.1.1.</li>
	 * </ul>
	 * 
	 * @return
	 */
	public String getVersionPath() {
		return String.format("%s/%s", getPath(), getVersionName().toString());
	}

	@Override
	public String getNamespace() {
		return repository.getNamespace();
	}

	@Override
	public String getOntologyFileName() {
		return repository.getOntologyFileName();
	}

	@Override
	public String getOntologyFileName(Languages l) {
		return repository.getOntologyFileName(l);
	}

	public void definesTerm(SAREFTerm term) {
		definedTerms.add(term);
	}

	public void usesTerm(SAREFTerm term) {
		usedTerms.add(term);
	}
	
	public Model getRequirements() {
		return requirements;
	}
	
	public Model getTests() {
		return tests;
	}

	@Override
	public String toString() {
		return "SAREFVersion " + repository.getProject() + " " + versionName;
	}

}
