/*
 * Copyright 2020 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.entities;

import java.nio.file.Path;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import fr.mines_stetienne.ci.saref.SAREF;
import fr.mines_stetienne.ci.saref.utils.Languages;

public class SAREFExample extends SAREFNamedGraph {

	public static final Pattern PATTERN_EXAMPLE_NAME = Pattern.compile(SAREF.REGEX_EXAMPLE_FILE_NAME);

	private final SAREFVersion version;
	private final String name;
	private final String fileName;
	private final String namespace;
	private final Set<SAREFTerm> exemplifiedTerms = new HashSet<>();
	
	private static String getName(String fileName) throws IllegalArgumentException {
		Matcher matcher = PATTERN_EXAMPLE_NAME.matcher(fileName);
		if (!matcher.find()) {
			throw new IllegalArgumentException();
		}
		return matcher.group(SAREF.REGEX_EXAMPLE_NAME_VAR);
	}
	
	private static String getIRI(SAREFVersion version, String fileName) throws IllegalArgumentException {
		String name = getName(fileName);
		return String.format("%sexample/%s#", version.getIRI(), name);
	}

	public SAREFExample(SAREFVersion version, String fileName) {
		super(getIRI(version, fileName));
		this.version = version;
		this.fileName = fileName;
		this.name = getName(fileName);
		this.namespace = String.format("%sexample/%s/", version.getIRI(), name);
	}

	public SAREFExample(SAREFVersion version, Path p) {
		this(version, p.toFile().getName());
	}

	public String getName() {
		return name;
	}

	public SAREFVersion getVersion() {
		return version;
	}

	public SAREFRepository getRepository() {
		return version.getRepository();
	}

	public String getFileName() {
		return fileName;
	}
	
	public String getFileName(Languages l) {
		return String.format("%s.%s", name, l.getExt());
	}

	public String getNamespace() {
		return namespace;
	}
	
	public void exemplifiesTerm(SAREFTerm term) {
		exemplifiedTerms.add(term);
	}
	
	public Set<SAREFTerm> getExemplifiedTerms() {
		return exemplifiedTerms;
	}

	@Override
	public String toString() {
		return String.format("SAREFExample %s %s %s", version.getRepository().getProject(), version.getVersionName(), name);
	}

}
