/*
 * Copyright 2024 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.checkers;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.nio.file.Files;

import fr.mines_stetienne.ci.saref.SAREFPipelineException;
import fr.mines_stetienne.ci.saref.managers.RepositoryManager;
import fr.mines_stetienne.ci.saref.utils.Languages;

/**
 * Checks TS 103 673 Clause 9.8.1: Vocabularies ontology directory. Need at least one file.
 * 
 */
public class Clause_9_8_1_Checker extends AbstractClauseChecker {

	private static enum MESSAGE {
		error, missing, ioexception, line, one, name, turtle;
	}

	public Clause_9_8_1_Checker(RepositoryManager repositoryManager) {
		super(repositoryManager, Clause_9_8_1_Checker.class);
	}

	@Override
	public void checkClause() throws SAREFPipelineException {
		File dir = new File(repository.getDirectory(), "vocabularies");
		if (!dir.isDirectory()) {
			return;
		}
		File[] fileList = null;

		try {
			if (Files.walk(dir.toPath(), 1).filter(p -> !p.toFile().isFile() && !p.toFile().getName().startsWith("."))
					.count() != 1) {
				logError(getMessage(Clause_9_8_1_Checker.MESSAGE.one, repository.getProject().getOntologyFileName(Languages.TEXT_TURTLE)));
			}
			// Collect the available vocabulary files.
			FileFilter filter = new FileFilter() {
				public boolean accept(File f)
				{
					return f.getName().endsWith("ttl");
				}
			};
			fileList = dir.listFiles(filter);
		} catch (IOException e) {
			logError(getMessage(Clause_9_8_1_Checker.MESSAGE.ioexception), e);
		}

		/*File file = new File(dir, repository.getProject().getOntologyFileName(Languages.TEXT_TURTLE));
		if (!file.isFile()) {
			String msg = getMessage(MESSAGE.name, repository.getOntologyFileName(Languages.TEXT_TURTLE));
			logError(msg);
			throw new SAREFPipelineException(msg);
		}*/
		for (File child : fileList) {
			System.out.println("!!! "+child.getName()+" !!!");//<<<
		}
		//<<< Check if saref core imports each vocabulary file. Iterate over each element in fileList and call checks().
		try {
			new Clause_9_8_1_1_Checker(repositoryManager).check();
			new Clause_9_8_1_2_Checker(repositoryManager).check();
			new Clause_9_8_1_3_Checker(repositoryManager).check();
			new Clause_9_8_1_4_Checker(repositoryManager).check();
			new Clause_9_8_1_5_Checker(repositoryManager).check();
		} catch (SAREFPipelineException ex) {
			logError(getMessage(Clause_9_8_1_Checker.MESSAGE.error), ex);
		}

	}

}
