/*
 * Copyright 2020 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.checkers;

import fr.mines_stetienne.ci.saref.SAREFPipelineException;
import fr.mines_stetienne.ci.saref.entities.SAREFVersion;
import fr.mines_stetienne.ci.saref.entities.SAREFVersionName;
import fr.mines_stetienne.ci.saref.managers.RepositoryManager;
import fr.mines_stetienne.ci.saref.vocabs.SHACL;

/**
 * Checks TS 103 673 Clause 9.4.3.1: Ontology IRI and Ontology Version IRI
 * 
 */
public class Clause_9_4_3_1_Checker extends AbstractShaclChecker {

	private static enum MESSAGE implements MessageResource {
		versionIRI, versionInfo, priorVersion_zero, priorVersion, imports, preferredNamespacePrefix,
		preferredNamespaceUri
	}

	public Clause_9_4_3_1_Checker(RepositoryManager repositoryManager) throws SAREFPipelineException {
		super(repositoryManager, Clause_9_4_3_1_Checker.class);
	}

	protected final void updateShapeModel() {
		add(exactly(versionName.toString()), MESSAGE.versionInfo, versionName.toString());
		add(exactly(version.getIRI()), MESSAGE.versionIRI, version.getIRI());
		SAREFVersionName priorVersionName = getPriorVersion(version);
		if (priorVersionName == null) {
			shapeModel.addLiteral(MESSAGE.priorVersion.asResource(), SHACL.maxCount, 0);
			shapeModel.add(MESSAGE.priorVersion.asResource(), SHACL.message,
					getMessage(MESSAGE.priorVersion_zero, version.getIRI()));
		} else {
			shapeModel.addLiteral(MESSAGE.priorVersion.asResource(), SHACL.minCount, 1);
			shapeModel.addLiteral(MESSAGE.priorVersion.asResource(), SHACL.maxCount, 1);
			add(exactly(repository.getIRI(priorVersionName)), MESSAGE.priorVersion, priorVersionName, versionName, repository.getIRI(priorVersionName));
		}
		add(MESSAGE.imports);
		add(exactly(repository.getPrefix()), MESSAGE.preferredNamespacePrefix, repository.getPrefix());
		add(exactly(repository.getNamespace()), MESSAGE.preferredNamespaceUri, repository.getNamespace());
		
		if (versionName.equals(SAREFVersionName.DEFAULT)) {
			remove(MESSAGE.versionInfo);
			remove(MESSAGE.versionIRI);
			remove(MESSAGE.priorVersion);
		}
	}

	private SAREFVersionName getPriorVersion(SAREFVersion version) {
		SAREFVersionName versionName = version.getVersionName();
		for (SAREFVersionName otherVersionName : version.getRepository().getVersions().descendingKeySet()) {
			if (otherVersionName.compareTo(versionName) < 0) {
				SAREFVersion otherVersion = version.getRepository().getVersions().get(otherVersionName);
				if (otherVersion.getReleaseRef() != null) {
					return otherVersionName;
				}
			}
		}
		return null;
	}
	
}