/*
 * Copyright 2020 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.entities;

import java.util.HashSet;
import java.util.Set;

import fr.mines_stetienne.ci.saref.SAREF;
import fr.mines_stetienne.ci.saref.utils.Languages;

/**
 *
 * @author maxime.lefrancois
 */
public class SAREFTerm extends SAREFNamedGraph {

	private final SAREFRepository repository;
	private final SAREFProject project;
	private final String localName;
	private final String termPath;
	private final Set<SAREFVersion> isDefinedBy = new HashSet<>();
	private final Set<SAREFVersion> isUsedBy = new HashSet<>();
	private final Set<SAREFExample> isExemplifiedBy = new HashSet<>();
	
	public SAREFTerm(SAREFRepository repository, String iri) {
		super(iri);
		this.repository = repository;
		this.project = repository.getProject();
		this.localName = iri.substring(repository.getNamespace().length());
		this.termPath = iri.substring(SAREF.BASE.length());
	}

	public SAREFProject getProject() {
		return project;
	}

	public String getPrefixedName() {
		return String.format("%s:%s", project.getPrefix(), localName);
	}

	public String getLocalName() {
		return localName;
	}

	public String getTermPath() {
		return termPath;
	}

	public SAREFRepository getRepository() {
		return repository;
	}

	public String getFileName(Languages l) {
		return String.format("%s.%s", getLocalName(), l.getExt());
	}

	public void isDefinedBy(SAREFVersion version) {
		isDefinedBy.add(version);
	}

	public void isUsedBy(SAREFVersion version) {
		isUsedBy.add(version);
	}

	public Set<SAREFVersion> getIsDefinedBy() {
		return isDefinedBy;
	}	
	
	public Set<SAREFVersion> getIsUsedBy() {
		return isUsedBy;
	}
	
	public Set<SAREFExample> getIsExemplifiedBy() {
		return isExemplifiedBy;
	}
	
	public boolean isUsed() {
		return !isUsedBy.isEmpty();
	}
	
	public boolean isDefined() {
		return !isDefinedBy.isEmpty();
	}
	
	public boolean isExemplified() {
		return !isExemplifiedBy.isEmpty();
	}

	@Override
	public String toString() {
		return String.format("SAREFTerm %s:%s", project.getPrefix(), localName);
	}

	public void isExemplifiedBy(SAREFExample example) {
		isExemplifiedBy.add(example);
	}

}
