/*
 * Copyright 2020 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.checkers;

import org.eclipse.jgit.lib.Ref;
import org.slf4j.Logger;

import fr.mines_stetienne.ci.saref.SAREFPipelineException;
import fr.mines_stetienne.ci.saref.managers.RepositoryManager;

/**
 * Checks that develop branch is up to date with prerelease
 * branch, and prerelease branch is up to date with release branch.
 * 
 */
public class VersionsRefChecker extends AbstractChecker {

	private static enum MESSAGE {
		different;
	}

	/**
	 * @param isTarget    if true, then local branches are read. Else, remote
	 *                    branches are read.
	 */
	public VersionsRefChecker(RepositoryManager repositoryManager, Logger errorLogger) {
		super(repositoryManager, errorLogger);
	}

	@Override
	public void check() throws SAREFPipelineException {
		final Ref developRef = version.getDevelopRef();
		final Ref prereleaseRef = version.getPrereleaseRef();
		final Ref releaseRef = version.getReleaseRef();
		final String developId = developRef != null ? developRef.getObjectId().getName() : null;
		final String prereleaseId = prereleaseRef != null ? prereleaseRef.getObjectId().getName() : null;
		final String releaseId = releaseRef != null ? releaseRef.getObjectId().getName() : null;
		if (developId != null && prereleaseId != null && !developId.equals(prereleaseId)) {
			final String msg = String.format(getMessage(MESSAGE.different), "`develop`", "`prerelease`",
					versionName);
			errorLogger.warn(msg);
		}
		if (developId != null && releaseId != null && !developId.equals(releaseId)) {
			final String msg = String.format(getMessage(MESSAGE.different), "`develop`", "`release`",
					versionName);
			errorLogger.warn(msg);
		}
		if (prereleaseId != null && releaseId != null && !prereleaseId.equals(releaseId)) {
			final String msg = String.format(getMessage(MESSAGE.different), "`prerelease`", "`release`",
					versionName);
			errorLogger.warn(msg);
		}
	}

}
