/*
 * Copyright 2024 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.checkers;

import java.io.File;
import java.io.IOException;

import fr.mines_stetienne.ci.saref.SAREFPipelineException;
import fr.mines_stetienne.ci.saref.entities.SAREFExample;
import fr.mines_stetienne.ci.saref.managers.RepositoryManager;

/**
 * Checks TS 103 673 Clause 9.6: Ontology examples
 * 
 */
public class Clause_9_6_Checker extends AbstractClauseChecker {

	private static enum MESSAGE {
		directories
	};

	public Clause_9_6_Checker(RepositoryManager repositoryManager) {
		super(repositoryManager, Clause_9_6_Checker.class);
	}

	@Override
	public void checkClause() throws SAREFPipelineException {
		File dir = new File(repository.getDirectory(), "examples");
		if (!dir.isDirectory()) {
			//return;
			try {
				String[] lines = {
						"@prefix dcterms: <http://purl.org/dc/terms/> .",
						"@prefix dctype: <http://purl.org/dc/dcmitype/> .",
						"@prefix geo: <http://www.w3.org/2003/01/geo/wgs84_pos#> .",
						"@prefix geosp: <http://www.opengis.net/ont/geosparql#> .",
						"@prefix owl: <http://www.w3.org/2002/07/owl#> .",
						"@prefix rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#> .",
						"@prefix rdfs: <http://www.w3.org/2000/01/rdf-schema#> .",
						"@prefix saref: <https://saref.etsi.org/core/> .",
						"@prefix time: <http://www.w3.org/2006/time#> .",
						"@prefix xsd: <http://www.w3.org/2001/XMLSchema#> .",
						"### CHANGE abcd !!! ",
						"@prefix s4abcd: <https://saref.etsi.org/saref4abcd/> .",
						"@prefix ex: <https://saref.etsi.org/saref4abcd/v1.1.1/example/example1/> .",
						"",
						"<https://saref.etsi.org/saref4abcd/v1.1.1/example/example1#> a dctype:Dataset ;",
						"dcterms:license <https://forge.etsi.org/etsi-software-license> ;",
						"dcterms:conformsTo <https://saref.etsi.org/core/v3.2.1/> ;",
						"dcterms:conformsTo <https://saref.etsi.org/saref4abcd/v1.1.1/> ;",
						"dcterms:title \"SAREF4abcd example number 1.\"@en ;",
						"dcterms:description \"SAREF4abcd example number 1.\"@en .",
						"",
				};
				CreateFileInDirectoryWithContents("examples", "example1.ttl", lines);
			} catch (IOException ex) {
				logError(getMessage(Clause_9_6_Checker.MESSAGE.directories));
			}
		}
		
		new Clause_9_6_1_Checker(repositoryManager).check();
		
		for(SAREFExample example: version.getExamples().values()) {
			new Clause_9_6_2_Checker(repositoryManager, example).check();
			new Clause_9_6_3_Checker(repositoryManager, example).check();
			new Clause_9_6_4_Checker(repositoryManager, example).check();
		}
	}
	
}
