/*
 * Copyright 2020 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.Map;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;

import org.apache.commons.io.FileUtils;
import org.apache.log4j.Layout;
import org.apache.log4j.PatternLayout;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import fr.mines_stetienne.ci.saref.managers.DatasetManager;
import fr.mines_stetienne.ci.saref.managers.OntologyManager;
import fr.mines_stetienne.ci.saref.managers.SiteManager;
import fr.mines_stetienne.ci.saref.managers.SourcesManager;
import fr.mines_stetienne.ci.saref.tests.TestSuites;
import fr.mines_stetienne.ci.saref.tests.TestSuitesAppender;

public class SAREFPipeline {

	private static final Logger LOG = LoggerFactory.getLogger(SAREFPipeline.class);

	public static final String NAME_TARGET = "target";
	private static final String NAME_LOG_FILE = "output.log";

	public static enum Mode {
		DEVELOP, RELEASE, PRERELEASE_PORTAL, RELEASE_PORTAL, CLEAN
	}

	private final TestSuites testSuites = new TestSuites("SAREF pipeline");
	private final Map<String, Logger> LOGGER_MAP = new HashMap<>();

	public final SAREFCredentialsProvider credentialsProvider;
	public final File directory;
	public final File targetDir;
	public final Mode mode;
	public final boolean ignoreExamples;
	public final boolean ignoreTerms;
	public final boolean ignoreSite;
	public final boolean ignoreDocumentation;
	
	private Logger logger;
	private DatasetManager datasetManager;
	private SiteManager siteManager;
	private SourcesManager sourcesManager;
	private OntologyManager ontologyManager;

	public SourcesManager getSourcesManager() {
		return sourcesManager;
	}

	public SiteManager getSiteManager() {
		return siteManager;
	}
	
	public DatasetManager getDatasetManager() {
		return datasetManager;
	}

	public OntologyManager getOntologyManager() {
		return ontologyManager;
	}

	public SAREFPipeline(SAREFCredentialsProvider credentialsProvider, File directory, Mode mode, boolean ignoreExamples, boolean ignoreTerms, boolean ignoreSite, boolean ignoreDocumentation) {
		this.credentialsProvider = credentialsProvider;
		this.directory = directory;
		targetDir = new File(directory, NAME_TARGET);
		this.mode = mode;
		this.ignoreExamples = ignoreExamples;
		this.ignoreTerms = ignoreTerms;
		this.ignoreSite = ignoreSite;
		this.ignoreDocumentation = ignoreDocumentation;
	}

	public TestSuites run() {
		LOG.info(SAREF.getMessage("start", directory));
		try {
			if (mode == Mode.CLEAN) {
				FileUtils.forceDelete(targetDir);
				return testSuites;
			}
			FileUtils.forceMkdir(targetDir);
			prepareLogFile();
		} catch (IOException ex) {
			LOG.error("IOException:", ex);
			return testSuites;
		}
		try {
			logger = getLogger(SAREF.getMessage("start", directory));

			ontologyManager = new OntologyManager(this, logger);

			siteManager = new SiteManager(this, logger);
			siteManager.prepareSite();
			
			sourcesManager = new SourcesManager(this, logger);
			sourcesManager.initTargetRepositoryManager();
			sourcesManager.fetchRepositories();
			sourcesManager.loadRepositories();
			sourcesManager.checkClauses();
			sourcesManager.checkTerms();
			sourcesManager.generateSite();
			sourcesManager.resetCheckout();
			if(mode == Mode.PRERELEASE_PORTAL || mode == Mode.RELEASE_PORTAL) {
				datasetManager = new DatasetManager(this, logger);
				datasetManager.createDataset();
			}
		} catch (IOException ex) {
			LOG.warn("IOException:", ex);
		} catch (SAREFPipelineException ex) {
			LOG.warn("SAREFPipelineException:", ex);
		}
		testSuites.clean();
		writeReport();
		return testSuites;
	}

	/**
	 * Prepare the log file in the target directory repository.
	 */
	private void prepareLogFile() throws IOException {
		final File logFile = new File(targetDir, NAME_LOG_FILE);
		final Layout layout = new PatternLayout("%-5p - %m%n");
		final org.apache.log4j.Logger rootLogger = org.apache.log4j.Logger.getRootLogger();
		rootLogger.addAppender(new org.apache.log4j.RollingFileAppender(layout, logFile.getAbsolutePath(), false));
		final org.apache.log4j.Logger loggerBase = org.apache.log4j.Logger.getLogger(SAREF.LOGGER_BASE);
		final TestSuitesAppender appender = new TestSuitesAppender(testSuites);
		loggerBase.addAppender(appender);
	}

	/**
	 * Prepare the log file in the target directory repository.
	 */
	public Logger getLogger(String testSuiteName) {
		Logger logger = LOGGER_MAP.get(testSuiteName);
		if (logger != null) {
			return logger;
		}
		logger = LoggerFactory.getLogger(SAREF.LOGGER_BASE + "." + testSuiteName);
		LOGGER_MAP.put(testSuiteName, logger);
		return logger;
	}

	private void writeReport() {
		try {
			File report = new File(targetDir, "report_output.xml");
			JAXBContext jaxbContext = JAXBContext.newInstance(TestSuites.class);

			Marshaller jaxbMarshaller = jaxbContext.createMarshaller();
			jaxbMarshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);
			jaxbMarshaller.marshal(testSuites, report);
			final StringWriter sw = new StringWriter();
			jaxbMarshaller.marshal(testSuites, sw);
			if (siteManager != null) {
				siteManager.writeReport(sw);
			}
		} catch (JAXBException | IOException ex) {
			LOG.error("Exception:", ex);
			ex.printStackTrace();
		}
	}

}
