/*
 * Copyright 2020 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.managers;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

import org.apache.jena.atlas.io.IndentedWriter;
import org.apache.jena.query.Dataset;
import org.apache.jena.query.DatasetFactory;
import org.apache.jena.query.QueryExecutionFactory;
import org.apache.jena.rdf.model.Model;
import org.apache.jena.rdf.model.ModelFactory;
import org.apache.jena.rdf.model.Property;
import org.apache.jena.rdf.model.Resource;
import org.apache.jena.vocabulary.OWL2;
import org.apache.jena.vocabulary.RDF;
import org.apache.jena.vocabulary.RDFS;
import org.semanticweb.owlapi.model.OWLAnnotationValue;
import org.semanticweb.owlapi.model.OWLLiteral;
import org.semanticweb.owlapi.model.OWLOntology;
import org.semanticweb.owlapi.model.parameters.Imports;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import fr.mines_stetienne.ci.saref.SAREF;
import fr.mines_stetienne.ci.saref.SAREFPipeline;
import fr.mines_stetienne.ci.saref.SAREFPipelineException;
import fr.mines_stetienne.ci.saref.SAREFRepositoryErrorLogger;
import fr.mines_stetienne.ci.saref.entities.SAREFRepository;
import fr.mines_stetienne.ci.saref.entities.SAREFTerm;
import fr.mines_stetienne.ci.saref.entities.SAREFVersion;
import fr.mines_stetienne.ci.saref.utils.Languages;
import fr.mines_stetienne.ci.saref.vocabs.EX;

public class TermSiteManager extends SAREFRepositoryErrorLogger {
	
	private static final Logger LOG = LoggerFactory.getLogger(TermSiteManager.class);

	private static final String REPO = "repo";

	private final static Logger getLogger(SAREFPipeline pipeline, SAREFRepository repository) {
		String repositoryName = repository.getName();
		final String loggerName = SAREF.getMessage(REPO, repositoryName);
		return pipeline.getLogger(loggerName);
	}

	private static enum MESSAGE {
		term_rdf, term_html
	}

	private final File siteDir;
	private final File ontoDir;
	
	
	public TermSiteManager(SAREFPipeline pipeline, SAREFRepository repository) {
		super(pipeline, repository, getLogger(pipeline, repository));
		this.siteDir = new File(pipeline.targetDir, SAREF.NAME_SITE);
		this.ontoDir = new File(siteDir, repository.getPath());
	}
	
	public void generateSite() throws SAREFPipelineException {
		for(SAREFTerm term: repository.getTerms().values()) {
			LOG.info("Generating site for " + term);
			generateModel(term);
			generateRDFFiles(term);
			generateHTMLFile(term);
		}
	}

	private void generateModel(SAREFTerm term) throws SAREFPipelineException {
		Model model = term.getModel();
		model.add(term.getResource(), RDF.type, RDFS.Resource);
		model.add(term.getResource(), EX.localName, term.getLocalName());
		term.getIsDefinedBy().stream().forEach(v->{
			model.add(term.getResource(), RDFS.isDefinedBy, v.getResource());
		});
		term.getIsUsedBy().stream().forEach(v->{
			model.add(term.getResource(), EX.isUsedBy, v.getResource());
		});		
		SAREFVersion lastVersion = getLastVersion(term);
		if(lastVersion == null) {
			return;
		}
		model.add(QueryExecutionFactory.create(String.format("DESCRIBE <%s>", term.getIRI()), lastVersion.getModel()).execDescribe());
	}

	private SAREFVersion getLastVersion(SAREFTerm term) {
		SAREFVersion lastVersion = null;
		for(SAREFVersion v : term.getIsDefinedBy()) {
			if(lastVersion == null) {
				lastVersion = v;
				continue;
			}
			if(v.getVersionName().compareTo(lastVersion.getVersionName()) > 0) {
				lastVersion = v;
			}
		}
		return lastVersion;
	}

	private void generateRDFFiles(SAREFTerm term) throws SAREFPipelineException {
		Model model = term.getModel();
		for (Languages l : Languages.values()) {
			String termFileName = term.getFileName(l);
			File file = new File(ontoDir, termFileName);
			try (FileOutputStream fos = new FileOutputStream(file)) {
				model.write(fos, l.getLang());
			} catch (IOException ex) {
				String msg = getMessage(MESSAGE.term_rdf, project, term.getLocalName());
				errorLogger.warn(msg, ex);
			}
		}		
	}

	private void generateHTMLFile(SAREFTerm term) throws SAREFPipelineException {
		Dataset dataset = DatasetFactory.create();
		dataset.addNamedModel(term.getIRI(), term.getModel());
		SAREFVersion lastVersion = term.getRepository().getVersions().lastEntry().getValue();
		Model lastVersionModel = ModelFactory.createDefaultModel();
		lastVersionModel.add(lastVersion.getModel());
		OWLOntology ontology = ontologyManager.loadOntology(lastVersion, errorLogger);
		ontology.classesInSignature(Imports.INCLUDED).forEach(c->{
			if(ontology.isDeclared(c)) {
				return;
			}
			Resource r = lastVersionModel.getResource(c.getIRI().toString());
			lastVersionModel.add(r, RDF.type, OWL2.Class);
			ontology.annotationAssertionAxioms(c.getIRI(), Imports.INCLUDED).forEach(axiom->{
				Property p = lastVersionModel.getProperty(axiom.getProperty().getIRI().toString());
				OWLAnnotationValue value = axiom.getValue();
				if(value instanceof OWLLiteral) {
					OWLLiteral literal = (OWLLiteral) value;
					lastVersionModel.add(r, p, literal.getLiteral());
				}
				
			});;
		});		
		ontology.objectPropertiesInSignature(Imports.INCLUDED).forEach(op->{
			if(ontology.isDeclared(op)) {
				return;
			}
			Resource r = lastVersionModel.getResource(op.getIRI().toString());
			lastVersionModel.add(r, RDF.type, OWL2.ObjectProperty);
			ontology.annotationAssertionAxioms(op.getIRI(), Imports.INCLUDED).forEach(axiom->{
				Property p = lastVersionModel.getProperty(axiom.getProperty().getIRI().toString());
				OWLAnnotationValue value = axiom.getValue();
				if(value instanceof OWLLiteral) {
					OWLLiteral literal = (OWLLiteral) value;
					lastVersionModel.add(r, p, literal.getLiteral());
				}
				
			});;
		});
		ontology.dataPropertiesInSignature(Imports.INCLUDED).forEach(dp->{
			if(ontology.isDeclared(dp)) {
				return;
			}
			Resource r = lastVersionModel.getResource(dp.getIRI().toString());
			lastVersionModel.add(r, RDF.type, OWL2.DatatypeProperty);
			ontology.annotationAssertionAxioms(dp.getIRI(), Imports.INCLUDED).forEach(axiom->{
				Property p = lastVersionModel.getProperty(axiom.getProperty().getIRI().toString());
				OWLAnnotationValue value = axiom.getValue();
				if(value instanceof OWLLiteral) {
					OWLLiteral literal = (OWLLiteral) value;
					lastVersionModel.add(r, p, literal.getLiteral());
				}
				
			});;
		});
		ontology.individualsInSignature(Imports.INCLUDED).forEach(i->{
			if(ontology.isDeclared(i)) {
				return;
			}
			Resource r = lastVersionModel.getResource(i.getIRI().toString());
			lastVersionModel.add(r, RDF.type, OWL2.NamedIndividual);
			ontology.annotationAssertionAxioms(i.getIRI(), Imports.INCLUDED).forEach(axiom->{
				Property p = lastVersionModel.getProperty(axiom.getProperty().getIRI().toString());
				OWLAnnotationValue value = axiom.getValue();
				if(value instanceof OWLLiteral) {
					OWLLiteral literal = (OWLLiteral) value;
					lastVersionModel.add(r, p, literal.getLiteral());
				}
				
			});;
		});
		
		dataset.addNamedModel(lastVersion.getIRI(), lastVersionModel);
		
		String htmlFileName = String.format("%s.html", term.getLocalName());
		File file = new File(ontoDir, htmlFileName);
		try (IndentedWriter writer = new IndentedWriter(new FileOutputStream(file))) {
			siteManager.generateTermDocumentation(term, writer, dataset);
		} catch (Exception ex) {
			String msg = getMessage(MESSAGE.term_html, project, term.getLocalName());
			errorLogger.warn(msg, ex);
		}		
	}
	
}
