/*
 * Copyright 2024 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.managers;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URL;
import java.net.URLEncoder;
import java.net.http.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.Base64;

import com.fasterxml.jackson.databind.ObjectMapper;
import fr.mines_stetienne.ci.saref.SAREFPipelineException;

/*	docker run -d --name SAREFvalidator -p 8080:8080 isaitb/shacl-validator	 # runs the generic validator.
	Docs for RDF validator: https://www.itb.ec.europa.eu/docs/guides/latest/validatingRDF/index.html
	Base docker image: https://hub.docker.com/r/isaitb/shacl-validator  (generic validator)
	Use Swagger UI for testing: http://localhost:8080/shacl/swagger-ui/index.html#/{domain}/api/validate
	BASE64 Encoder/Decoder: https://www.base64encode.org/
* */
public class ShaclValidationManager {

	private static String VALIDATOR_URL = "http://localhost:8080/shacl/";
	private String ontologyToValidate;	// file:///home/davidgnabasik/dev/real_saref4auto/ontology/saref4auto.ttl or URL
	private HttpClient client;
	private ObjectMapper objectMapper;

	//private String shaclRequestBody =
	//		"{ \"contentToValidate\": \"ONTOLOGY_AS_BASE64_STRING\", \"embeddingMethod\": \"BASE64\", \"contentSyntax\": \"text/turtle\", \"reportSyntax\": \"application/json\", \"locale\": \"en\", \"rdfReportSyntax\": \"application/json\", \"externalRules\": [ { \"ruleSet\": \"SHACL_AS_BASE64_STRING\", \"embeddingMethod\": \"BASE64\", \"ruleSyntax\": \"text/turtle\" } ] }";

	private static String encodeBody(Object obj) {
		return URLEncoder.encode(obj.toString(), StandardCharsets.UTF_8);
	}

	private enum MESSAGE {	// each of these accepts a single string parameter.
		http_exception, ioexception
	}

	/**
	 * Open a REST interface to the SHACL validator.
	 */
	public ShaclValidationManager(String ontologyToValidate) throws IOException {
		client = HttpClient.newHttpClient();
		objectMapper = new ObjectMapper();
		this.ontologyToValidate = ontologyToValidate;
		String url = System.getenv("VALIDATOR_URL");
		if (url != null)  VALIDATOR_URL = url;
	}

	/**
	 * Is the validation server accessible?
	 * @throws SAREFPipelineException
	 */
	public static void PingValidationServer() throws SAREFPipelineException {
		try {
			URL url = new URL(VALIDATOR_URL);
			HttpURLConnection connection = (HttpURLConnection) url.openConnection();
			connection.setRequestMethod("GET");
			connection.connect();
			connection.getResponseCode();
		} catch (IOException e) {
			throw new SAREFPipelineException("response", "Unknown host: " + VALIDATOR_URL);
		}
		//logError(getMessage(ShaclValidationManager.MESSAGE.http_exception, VALIDATOR_URL));
	}

	/**
	 * Preferred validation format is BASE64. Read file; return single BASE64-encoded string.
	 * @throws SAREFPipelineException
	 */
	public static String FormatAsBase64(String ontologyFile) throws SAREFPipelineException {
		try {
			String value = Files.readString(Path.of(ontologyFile));  // 2Gb file size limit.
			return Base64.getUrlEncoder().encodeToString(value.getBytes());
		} catch (IOException ex) {
			throw new SAREFPipelineException("FormatAsBase64", "Unable to format file as base64: " + ontologyFile);
		}
	}

	public static HttpRequest.BodyPublisher ofForm(Map<String, Object> data) {
		StringBuilder body = new StringBuilder();
		for (Object dataKey : data.keySet()) {
			if (body.length() > 0) {
				body.append("&");
			}
			body.append(encodeBody(dataKey))
					.append("=")
					.append(encodeBody(data.get(dataKey)));
		}
		return HttpRequest.BodyPublishers.ofString(body.toString());
	}

	/**
	 * POST JSON response format:
	 {
	 "date": "2024-04-05T10:18:24.197+0000",
	 "result": "SUCCESS",
	 "overview": {
	 "profileID": "Clause_9_4_3_1"
	 },
	 "counters": {
	 "nrOfAssertions": 0,
	 "nrOfErrors": 0,
	 "nrOfWarnings": 0
	 },
	 "context": {},
	 "reports": {},
	 "name": "SHACL Validation"
	 }

	 * Generic validator option: 1. Replace {contentToValidate} value with the ontology to validate encoded in BASE64. (This can aos be done with URLs.)
	 * 2. Configure the user-provided SHACL shape as a RuleSet element of the externalRules array. The content of each RuleSet is as follows:
	 * [ "ruleSet": "BASE64-encoded_SHACL_shape", "embeddingMethod":"BASE64", "ruleSyntax": "text/turtle" ]   We run one SHACL shape at a time.
	 * POST request to Validate a single RDF instance.
	 * @throws IOException
	 * @throws InterruptedException
	 */
	public HttpResponse<String> validateOntologyWithShacl(String shaclFile) throws SAREFPipelineException, InterruptedException {
		try {
			String ontologyBase64String = FormatAsBase64(this.ontologyToValidate);
			String shaclBase64String = FormatAsBase64(shaclFile);
			// populate shaclRequestBody json struct to map (no TypeReference)
			String shaclRequestBody = "{ \"contentToValidate\": \"" + ontologyBase64String +
				"\", \"embeddingMethod\": \"BASE64\", \"contentSyntax\": \"text/turtle\", \"reportSyntax\": \"application/json\", \"locale\": \"en\", \"rdfReportSyntax\": \"application/json\", \"externalRules\": [ { \"ruleSet\": \"" +
					shaclBase64String + "\", \"embeddingMethod\": \"BASE64\", \"ruleSyntax\": \"text/turtle\" } ] }";
			byte[] mapData = shaclRequestBody.getBytes();
			Map<String,Object> myMap = objectMapper.readValue(mapData, HashMap.class);
			// now update JSON data
			//myMap.put("contentToValidate", ontologyBase64String);
			//myMap.put("ruleSet", shaclBase64String);
			HttpRequest request = HttpRequest.newBuilder()
					.header("Content-Type", "text/turtle")    // application/x-www-form-urlencoded
					.uri(URI.create(VALIDATOR_URL + "any/upload"))
					.POST(ofForm(myMap))
					.build();

			HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());
			System.out.println("Status code: " + response.statusCode()); //<<<
			System.out.println("\n Body: " + response.body());
			return response;
		} catch (IOException ex) {
			throw new SAREFPipelineException("validateOntologyWithShacl", "Unable to process SHACL file: " + shaclFile);
		}
	}

}
