/*
 * Copyright 2024 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.checkers;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.net.http.HttpResponse;
import java.nio.file.Files;
import java.util.stream.Collectors;

import fr.mines_stetienne.ci.saref.SAREFPipelineException;
import fr.mines_stetienne.ci.saref.managers.RepositoryManager;
import fr.mines_stetienne.ci.saref.managers.ShaclValidationManager;
import fr.mines_stetienne.ci.saref.utils.Languages;
import org.semanticweb.owlapi.model.OWLOntology;

/**
 * Checks TS 103 673 Clause 9.4.6: Conformance to reference ontology patterns.
 * The ontology in the ontology document of a SAREF project version shall conform to the SHACL specification of each
 * reference ontology pattern defined in this SAREF project version.
 */
public class Clause_9_4_6_Checker extends AbstractClauseChecker {

	private enum MESSAGE {
		one, missing, ioexception, server_unavailable
	}

	public Clause_9_4_6_Checker(RepositoryManager repositoryManager) throws SAREFPipelineException {
		super(repositoryManager, Clause_9_4_6_Checker.class);
	}

	/**
	 The ontology in the ontology document of a SAREF project version shall conform to the SHACL specification of each
	 reference ontology pattern defined in this SAREF project version.
	 If the ontology in the ontology document of a SAREF project version imports an {Ontology | Vocabulary | Pattern} Version IRI defined in
	 another SAREF project version, either directly or transitively, then it shall conform to the SHACL specification of each
	 reference ontology pattern defined in this other SAREF project version.
	 This method calls the generic docker image that runs a specific shacl files.
	 * */
	@Override
	public void checkClause() throws SAREFPipelineException {
		OWLOntology ontology = pipeline.getOntologyManager().loadOntology(version, errorLogger);
		if (ontology == null) {
			return;
		}
		File dir = new File(repository.getDirectory(), "ontology");
		File file = new File(dir, repository.getProject().getOntologyFileName(Languages.TEXT_TURTLE));
		String ontologyToValidate = file.getAbsolutePath();

		// First get the available directories. If none, not a problem.
		File dir2 = new File(repository.getDirectory(), "patterns");
		String[] directoryList = new String[]{};
		try {
			String directories = Files.walk(dir2.toPath()).filter(p -> {
				try {
					return p.toFile().isDirectory() && !Files.isSameFile(dir2.toPath(), p);
				} catch(IOException ex) {
					return false;
				}
			}).map(p -> p.toString()).collect(Collectors.joining(", "));
			if (directories.isEmpty()) {
				return; // no SHACL directories found.
			}
			directoryList = directories.split(",");
		} catch (IOException e) {
			logError(getMessage(Clause_9_4_6_Checker.MESSAGE.ioexception), e);
		}

		ShaclValidationManager.PingValidationServer();

		Map<String, HttpResponse<String>> responseMap = new HashMap<String, HttpResponse<String>>();
		// process every shapes.ttl file in the sub-directories of the patterns directory.
		for(int i = 0; i < directoryList.length; i++) {
			String shaclFileName = directoryList[i] + "/shapes.ttl";  // "file://" +
			try {
				HttpResponse<String> response = new ShaclValidationManager(ontologyToValidate).validateOntologyWithShacl(shaclFileName);
				System.out.println(response); //<<<
				responseMap.put(shaclFileName, response);	//<<< parse map for errors
			} catch (IOException | InterruptedException e) {
				logError(getMessage(Clause_9_4_6_Checker.MESSAGE.ioexception, shaclFileName));
			}
		}
		//System.out.println(responseMap); //<<<
	}
}