/*
 * Copyright 2020 ETSI
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package fr.mines_stetienne.ci.saref.entities;

import java.io.File;
import java.util.HashMap;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;

import org.apache.jena.rdf.model.Resource;

import fr.mines_stetienne.ci.saref.utils.Languages;

public class SAREFRepository implements SAREFProject {

	private final File directory;
	private final SAREFProject project;
	private final NavigableMap<SAREFVersionName, SAREFVersion> versions = new TreeMap<>();
	private final Map<String, SAREFTerm> terms = new HashMap<>();
	private final boolean isClean;
	private final SAREFVersionName originalVersion;
	private final String originalBranch;

	public SAREFRepository(File directory, SAREFProject project, boolean isClean, SAREFVersionName originalVersion, String originalBranch) {
		this.directory = directory;
		this.project = project;
		this.isClean = isClean;
		this.originalVersion = originalVersion;
		this.originalBranch = originalBranch;
	}

	public File getDirectory() {
		return directory;
	}

	public NavigableMap<SAREFVersionName, SAREFVersion> getVersions() {
		return versions;
	}

	public SAREFVersion getVersion(SAREFVersionName versionName) {
		return versions.get(versionName);
	}

	public Map<String, SAREFTerm> getTerms() {
		return terms;
	}
	
	public String getOriginalBranch() {
		return originalBranch;
	}

	public SAREFVersionName getPreviousVersion(SAREFVersionName versionName) {
		return versions.floorKey(versionName);
	}
	
	public SAREFVersionName getOriginalVersion() {
		return originalVersion;
	}

	public boolean isClean() {
		return isClean;
	}

	/**
	 * Gets the IRI of this version
	 * @return
	 */
	public String getIRI(SAREFVersionName versionName) {
		return String.format("%s%s/", project.getNamespace(), versionName);
	}
	
	public SAREFProject getProject() {
		return project;
	}

	@Override
	public String getName() {
		return project.getName();
	}

	@Override
	public String getRepositoryName() {
		return project.getRepositoryName();
	}

	@Override
	public String getPrefix() {
		return project.getPrefix();
	}

	@Override
	public String getPath() {
		return project.getPath();
	}

	@Override
	public String getNamespace() {
		return project.getNamespace();
	}

	@Override
	public String getOntologyFileName() {
		return project.getOntologyFileName();
	}

	@Override
	public String getOntologyFileName(Languages l) {
		return project.getOntologyFileName(l);
	}

	public Resource getResource() {
		return project.getResource();
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((directory == null) ? 0 : directory.hashCode());
		result = prime * result + ((project == null) ? 0 : project.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		SAREFRepository other = (SAREFRepository) obj;
		if (directory == null) {
			if (other.directory != null)
				return false;
		} else if (!directory.equals(other.directory))
			return false;
		if (project == null) {
			if (other.project != null)
				return false;
		} else if (!project.equals(other.project))
			return false;
		return true;
	}

	@Override
	public String toString() {
		return "SAREFRepository " + project;
	}

}
