package org.etsi.osl.util;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;
import java.util.stream.Stream;

import org.etsi.osl.tmf.common.model.service.ServiceSpecificationRef;
import org.etsi.osl.tmf.lcm.model.LCMRuleSpecification;
import org.etsi.osl.tmf.rcm634.model.*;
import org.etsi.osl.tmf.scm633.model.ServiceSpecRelationship;
import org.etsi.osl.tmf.scm633.model.ServiceSpecification;
import org.etsi.osl.tmf.scm633.model.ServiceSpecificationCreate;
import org.etsi.osl.tmf.scm633.model.ServiceSpecificationUpdate;
import org.etsi.osl.tmf.lcm.model.LCMRuleSpecificationCreate;

import java.io.File;

public class ServiceSpecificationUploader {
    private static final Logger logger = LoggerFactory.getLogger(ServiceSpecificationUploader.class);
    private static final HttpClient client = HttpClient.newHttpClient();
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private static String apiEndpoint = null;
    private static final Map<String, String> uuidmap = new HashMap<>();
    private static final Map<String, String> idmap = new HashMap<>();
    private static final Map<String, String> uuid2nameversion = new HashMap<>();
    private static String configFilePath = null;
    private static KeycloakAuthenticator keycloakAuthenticator = null;
    private static final Properties config = new Properties();

    public static void main(String[] args) throws IOException {
        String serviceSpecFolderPath = null;
        // Parse command-line arguments
        for (int i = 0; i < args.length; i++) {
            switch (args[i]) {
                case "--configfile":
                    if (i + 1 < args.length) {
                        configFilePath = args[++i];
                        logger.info("Config file path set to: " + configFilePath);
                        try (InputStream input = new FileInputStream(configFilePath)) {
                            config.load(input);
                            apiEndpoint = config.getProperty("targetApiEndpoint.url");
                            logger.info("Configuration loaded successfully from " + configFilePath);
                        } catch (IOException ex) {
                            logger.error("Failed to load configuration from " + configFilePath, ex);
                            throw ex;
                        }
                        keycloakAuthenticator= new KeycloakAuthenticator(config);
                    } else {
                        logger.error("Missing value for --configfile");
                        printUsageAndExit();
                    }
                    break;
                case "--servicespecfolderpath":
                    if (i + 1 < args.length) {
                        serviceSpecFolderPath = args[++i];
                        logger.info("Service spec folder path set to: " + serviceSpecFolderPath);
                    } else {
                        logger.error("Missing value for --servicespecfolderpath");
                        printUsageAndExit();
                    }
                    break;
                default:
                    logger.error("Unknown argument: " + args[i]);
                    printUsageAndExit();
            }
        }

        // Validate required arguments
        if (configFilePath == null || serviceSpecFolderPath == null) {
            logger.error("Missing required arguments.");
            printUsageAndExit();
        }

        // Load and process the service specifications
        File serviceSpecDir = new File(serviceSpecFolderPath);
        if (!serviceSpecDir.exists() || !serviceSpecDir.isDirectory()) {
            logger.error("Service specification folder does not exist or is not a directory.");
            System.exit(1);
        }

        File[] serviceSpecFiles = serviceSpecDir.listFiles();
        if (serviceSpecFiles != null) {
            logger.info("Service Specification Files:");
            for (File file : serviceSpecFiles) {
                if (file.isFile()) {
                    logger.info("- " + file.getName());
                }
            }
        } else {
            logger.error("Unable to list files in the service specification folder.");
            System.exit(1);
        }

        // Continue with the rest of your application logic
        logger.info("Application started successfully with provided configurations.");

    	//String jsonFilesPath = "C:/openslice/servicespecificationfetcher/e83a4e2e-7036-44fa-8bd2-e0ddc954c63d"; //env.getProperty("json.files.path");
        logger.info("JSON Files Path: {}", serviceSpecFolderPath);
		Path initialPath = Paths.get(serviceSpecFolderPath);
        // Set the YAML path from another class or context
        logger.info("mappings YAML Files Path: {}", initialPath.resolve("mappings.yaml"));
        JsonStringReplacer.setYamlFilePath(initialPath.resolve("mappings.yaml").toString());
        // Example of processing the JSON files
        processServiceSpecificationRecursively(initialPath);
    }

    /**
     * Prints usage instructions and exits the application.
     */
    private static void printUsageAndExit() {
        String usage = "Usage:\n" +
                "  java -jar myapplication.jar --configfile /path/to/config.yaml --servicespecfolderpath /path/to/servicespec/\n\n" +
                "Parameters:\n" +
                "  --configfile               Path to the configuration file.\n" +
                "  --servicespecfolderpath    Path to the service specification folder.";
        logger.info(usage);
        System.exit(1);
    }

    private static void processServiceSpecificationRecursively(Path directoryPath) {

        logger.info("Going for ServiceRelationships Recursively at {}", directoryPath);
    	// Iterate for all subfolders in serviceSpecificationServiceRelationships
        try (Stream<Path> serviceSpecificationServiceRelationshipsPaths = Files.list(directoryPath.resolve("serviceSpecificationServiceRelationships"))) {
        	// As long as we find serviceSpecificationServiceRelationships we continue in depth
        	serviceSpecificationServiceRelationshipsPaths
                .filter(Files::isDirectory)
                .forEach(serviceSpecificationServiceRelationshipsPathSubDir -> processServiceSpecificationRecursively(serviceSpecificationServiceRelationshipsPathSubDir));
            
            // As long as we don't find serviceSpecificationServiceRelationships we continue with ResourceRelationships
            logger.info("Going for ResourceRelationships Recursively at {}", directoryPath);
            try (Stream<Path> serviceSpecificationResourceRelationshipsPaths = Files.list(directoryPath.resolve("serviceSpecificationResourceRelationships"))) {
            	serviceSpecificationResourceRelationshipsPaths
	            .filter(Files::isDirectory)
	            .forEach(serviceSpecificationResourceRelationshipsSubDir -> processResourceRelationshipsRecursively(serviceSpecificationResourceRelationshipsSubDir));            
            } catch (IOException e) {
                logger.error("Failed to read directory: {} - {}", directoryPath, e.getMessage());
            }

            // Process JSON files in the current directory after processing subdirectories
            logger.info("Going for JSON files at {}", directoryPath);
            try (Stream<Path> files = Files.list(directoryPath)) {
                files
                    .filter(Files::isRegularFile)
                    .filter(path -> path.toString().endsWith(".json") && !path.toString().endsWith("-lcmRuleIds.json"))
                    .forEach(path -> {
                        logger.info("Posting serviceSpecificationServiceRelationships json info at path {}", path);
                        postServiceSpec(path);
                    });
            }
        } catch (IOException e) {
            logger.error("Failed to read directory: {} - {}", directoryPath, e.getMessage());
        }
    }
    private static void processResourceRelationshipsRecursively(Path directoryPath) {
        logger.info("Going for ResourceRelationships Recursively at {}", directoryPath);
        // As long as we find resourceSpecificationResourceRelationships we continue in depth
        String appendSubpath = "serviceSpecificationResourceRelationships";
        if(directoryPath.toString().contains("serviceSpecificationResourceRelationships"))
        {
            appendSubpath="resourceSpecificationResourceRelationships";
        }

        try (Stream<Path> paths = Files.list(directoryPath.resolve(appendSubpath))) {
            paths
                    .filter(Files::isDirectory)
                    .forEach(subDir -> processResourceRelationshipsRecursively(subDir));

            // As long as we don't find resourceSpecificationResourceRelationships we continue with ResourceRelationship installation
            try (Stream<Path> files = Files.list(directoryPath)) {
                files
                        .filter(Files::isRegularFile)
                        .filter(path -> path.toString().endsWith(".json"))
                        .forEach(path -> {
                            logger.info("Working for ResourceSpecification on {}", path);
                            ObjectMapper objectMapper = new ObjectMapper();
                            ResourceSpecification resSpec = null;
                            JsonNode rootNode = null;
                            try {
                                rootNode = objectMapper.readTree(path.toFile());
                            } catch (IOException e) {
                                e.printStackTrace();
                            }
                            JsonNode typeNode = rootNode.get("@type");
                            String type = typeNode.asText();
                            if ("LogicalResourceSpecification".equals(type)) {
                                logger.info("The JSON contains a LogicalResourceSpecification.");
                                postLogicalResourceSpec(path);
                            } else if ("PhysicalResourceSpecification".equals(type)) {
                                logger.info("The JSON contains a PhysicalResourceSpecification.");
                                postPhysicalResourceSpec(path);
                            }
                            else {
                                logger.error("The JSON contains an unknown type: {}", type);
                            }
                        });
            }

        } catch (IOException e) {
            logger.error("Failed to read directory: {} - {}", directoryPath, e.getMessage());
        }
    }

    private static HttpResponse<String> sendHttpRequestWithRetry(String url, String token, String jsonContent, String method, String methodName, int maxRetries) throws IOException, InterruptedException {
        HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(url))
                .header("Content-Type", "application/json")
                .header("Authorization", "Bearer " + token)
                .method(method, HttpRequest.BodyPublishers.ofString(jsonContent))
                .build();

        int attempt = 0;
        HttpResponse<String> response;
        do {
            response = client.send(request, HttpResponse.BodyHandlers.ofString());
            if (response.statusCode() == 200) {
                return response;
            }
            attempt++;
        } while (attempt < maxRetries);

        throw new IOException("Failed request with status code: " + response.statusCode());
    }

    public static List<ResourceSpecification> getResourceSpecificationsInOpenslice() throws IOException, InterruptedException {
        String url = apiEndpoint + "/resourceCatalogManagement/v4/resourceSpecification";
        HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(url))
                .GET()
                .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() == 200) {
            List<ResourceSpecification> resourceSpecifications = new ArrayList<>();
            JsonNode rootNode = objectMapper.readTree(response.body());

            for (JsonNode node : rootNode) {
                String type = node.get("@type").asText();
                ResourceSpecification resourceSpecification;

                if ("LogicalResourceSpecification".equals(type)) {
                    resourceSpecification = objectMapper.treeToValue(node, LogicalResourceSpecification.class);
                } else if ("PhysicalResourceSpecification".equals(type)) {
                    resourceSpecification = objectMapper.treeToValue(node, PhysicalResourceSpecification.class);
                } else {
                    throw new IllegalArgumentException("Unsupported ResourceSpecification type: " + type);
                }
                resourceSpecifications.add(resourceSpecification);
            }
            return resourceSpecifications;
        } else {
            throw new IOException("Failed to get service specifications. HTTP status code: " + response.statusCode());
        }
    }

    public static List<ServiceSpecification> getServiceSpecificationsInOpenslice() throws IOException, InterruptedException {
        String url = apiEndpoint + "/serviceCatalogManagement/v4/serviceSpecification";
        HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(url))
                .GET()
                .build();

        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() == 200) {
            return objectMapper.readValue(response.body(), new TypeReference<List<ServiceSpecification>>() {});
        } else {
            throw new IOException("Failed to get service specifications. HTTP status code: " + response.statusCode());
        }
    }

    private static void postServiceSpec(Path path) {
    	try {
            ServiceSpecification servSpecFromFile = objectMapper.readValue(path.toFile(), ServiceSpecification.class);
            // If the ServiceSpecification name and version already exists in the database, skip it to avoid duplicates
            if (uuid2nameversion.containsValue(servSpecFromFile.getName() + " " + servSpecFromFile.getVersion())) {
                logger.info("ServiceSpecification with name {} and version {} already exists. Skipping.", servSpecFromFile.getName(), servSpecFromFile.getVersion());
                return;
            }
            // Use getServiceSpecificationsInOpenslice to get the available ServiceSpecifications in OpenSlice and chech if the servSpecFromFile exist in OPenslice. Check by comparing name and version.
            List<ServiceSpecification> serviceSpecificationsInOpenslice = getServiceSpecificationsInOpenslice();
            for (ServiceSpecification serviceSpecificationInOpenSlice : serviceSpecificationsInOpenslice) {
                if (serviceSpecificationInOpenSlice.getName().equals(servSpecFromFile.getName()) && serviceSpecificationInOpenSlice.getVersion().equals(servSpecFromFile.getVersion())) {
                    uuidmap.put(servSpecFromFile.getUuid(), serviceSpecificationInOpenSlice.getUuid());
                    idmap.put(servSpecFromFile.getId(), serviceSpecificationInOpenSlice.getId());
                    uuid2nameversion.put(serviceSpecificationInOpenSlice.getUuid(), serviceSpecificationInOpenSlice.getName() + " " + serviceSpecificationInOpenSlice.getVersion());
                    logger.info("ServiceSpecification with name {} and version {} already exists. Skipping.", servSpecFromFile.getName(), servSpecFromFile.getVersion());
                    return;
                }
            }

            logger.info("ServiceSpecification posting main properties.");
            ServiceSpecification newServSpec = postServiceSpecMainProperties(servSpecFromFile);
            uuidmap.put(servSpecFromFile.getUuid(), newServSpec.getUuid());
            idmap.put(servSpecFromFile.getId(), newServSpec.getId());
            uuid2nameversion.put(newServSpec.getUuid(), newServSpec.getName() + " " + newServSpec.getVersion());
            newServSpec = patchServiceSpecificationCharacteristics(servSpecFromFile, newServSpec);
            postLCMRules(path, newServSpec);
            postAttachment(path, newServSpec);
            newServSpec = patchServiceSpecificationServiceRelationships(servSpecFromFile, newServSpec);
            newServSpec = patchServiceSpecificationResourceRelationships(servSpecFromFile, newServSpec);
        } catch (IOException | InterruptedException e) {
            logger.info("Process halted due to failure: {}", e.getMessage());
        }
	}

    private static ResourceSpecification patchResourceSpecificationResourceRelationships(ResourceSpecification resourceSpecFromFile, ResourceSpecification resourceSpecToPatch) throws IOException, InterruptedException {
        // Check if servSpecUpdate has serviceSpecRelationship. If there is no serviceSpecRelationship, skip the patch
        if (resourceSpecFromFile.getResourceSpecRelationship() == null || resourceSpecFromFile.getResourceSpecRelationship().isEmpty()) {
            return resourceSpecToPatch;
        }
        ResourceSpecificationUpdate newResourceSpecUpdate = new ResourceSpecificationUpdate();
        List<ResourceSpecificationRelationship> newResourceSpecRelationship = new ArrayList<>();
        //Iterate over servSpecUpdate and replace uuids with the new ones
        for (ResourceSpecificationRelationship rsr : resourceSpecFromFile.getResourceSpecRelationship()) {
            if (idmap.containsKey(rsr.getId())) {
                rsr.setId(idmap.get(rsr.getId()));
                rsr.setUuid(uuidmap.get(rsr.getId()));
            }
            else {
                logger.error("Failed to find id: {}", rsr.getId());
            }

            newResourceSpecRelationship.add(rsr);
        }
        newResourceSpecUpdate.setResourceSpecificationRelationship(newResourceSpecRelationship);
        String jsonContent = JsonStringReplacer.replaceJsonStrings(objectMapper.writeValueAsString(newResourceSpecUpdate));
        String ruleUrl = apiEndpoint + "/resourceCatalogManagement/v4/resourceSpecification/" + resourceSpecToPatch.getUuid();
        String token = keycloakAuthenticator.getToken();

        HttpResponse<String> response = sendHttpRequestWithRetry(ruleUrl, token, jsonContent, "PATCH", "patch Resource Specification", 3);

        if (response.statusCode() != 200) {
            throw new IOException("Failed request with status code: " + response.statusCode());
        } else {
            JsonNode rootNode = objectMapper.readTree(response.body());
            String baseType = rootNode.get("@type").asText();
            if ("LogicalResourceSpecification".equals(baseType)) {
                return objectMapper.treeToValue(rootNode, LogicalResourceSpecification.class);
            } else if ("PhysicalResourceSpecification".equals(baseType)) {
                return objectMapper.treeToValue(rootNode, PhysicalResourceSpecification.class);
            } else {
                throw new IllegalArgumentException("Unsupported ResourceSpecification baseType: " + baseType);
            }
        }
    }

    private static ServiceSpecification patchServiceSpecificationServiceRelationships(ServiceSpecification servSpecFromFile, ServiceSpecification servSpecToPatch) throws IOException, InterruptedException {
        // Check if servSpecUpdate has serviceSpecRelationship. If there is no serviceSpecRelationship, skip the patch
        if (servSpecFromFile.getServiceSpecRelationship() == null || servSpecFromFile.getServiceSpecRelationship().isEmpty()) {
            return servSpecToPatch;
        }
        ServiceSpecificationUpdate newServiceSpecUpdate = new ServiceSpecificationUpdate();
        List<ServiceSpecRelationship> newServiceSpecRelationship = new ArrayList<>();
        //Iterate over servSpecUpdate and replace uuids with the new ones
        for (ServiceSpecRelationship ssr : servSpecFromFile.getServiceSpecRelationship()) {
            if (idmap.containsKey(ssr.getId())) {
                ssr.setId(idmap.get(ssr.getId()));
                ssr.setUuid(uuidmap.get(ssr.getId()));
            }
            else {
                logger.error("Failed to find id: {}", ssr.getId());
            }

            newServiceSpecRelationship.add(ssr);
        }
        newServiceSpecUpdate.setServiceSpecRelationship(newServiceSpecRelationship);
        String jsonContent = JsonStringReplacer.replaceJsonStrings(objectMapper.writeValueAsString(newServiceSpecUpdate));
        String ruleUrl = apiEndpoint + "/serviceCatalogManagement/v4/serviceSpecification/" + servSpecToPatch.getUuid();
        String token = keycloakAuthenticator.getToken();

        HttpResponse<String> response = sendHttpRequestWithRetry(ruleUrl, token, jsonContent, "PATCH", "patch Service Specification", 3);

        if (response.statusCode() != 200) {
            throw new IOException("Failed request with status code: " + response.statusCode());
        } else {
            return objectMapper.readValue(response.body(), ServiceSpecification.class);
        }
    }

    private static ServiceSpecification patchServiceSpecificationResourceRelationships(ServiceSpecification servSpecFromFile, ServiceSpecification servSpecToPatch) throws IOException, InterruptedException {
        // Check if servSpecUpdate has resourceSpecRelationship. If there is no resourceSpecRelationship, skip the patch
        if (servSpecFromFile.getResourceSpecification() == null || servSpecFromFile.getResourceSpecification().isEmpty()) {
            return servSpecToPatch;
        }
        ServiceSpecificationUpdate newServiceSpecUpdate = new ServiceSpecificationUpdate();
        List<ResourceSpecificationRef> newServiceSpecResourceRelationship = new ArrayList<>();
        //Iterate over servSpecUpdate and replace uuids with the new ones
        for (ResourceSpecificationRef rsr : servSpecFromFile.getResourceSpecification()) {
            if (idmap.containsKey(rsr.getId())) {
                rsr.setId(idmap.get(rsr.getId()));
                rsr.setUuid(uuidmap.get(rsr.getId()));
            }
            else {
                logger.error("Failed to find id: {}", rsr.getId());
            }

            newServiceSpecResourceRelationship.add(rsr);
        }
        newServiceSpecUpdate.setResourceSpecification(newServiceSpecResourceRelationship);
        String jsonContent = JsonStringReplacer.replaceJsonStrings(objectMapper.writeValueAsString(newServiceSpecUpdate));
        String ruleUrl = apiEndpoint + "/serviceCatalogManagement/v4/serviceSpecification/" + servSpecToPatch.getUuid();
        String token = keycloakAuthenticator.getToken();

        HttpResponse<String> response = sendHttpRequestWithRetry(ruleUrl, token, jsonContent, "PATCH", "patch Service Specification", 3);

        if (response.statusCode() != 200) {
            throw new IOException("Failed request with status code: " + response.statusCode());
        } else {
            return objectMapper.readValue(response.body(), ServiceSpecification.class);
        }
    }

    private static void postAttachment(Path path, ServiceSpecification newServSpec) {
        Path basePath = Files.isDirectory(path) ? path : path.getParent();
        logger.info("BasePath for attachments: {}", basePath);
        Path attachmentsPath = basePath.resolve("attachment");
        if (!Files.exists(attachmentsPath) || !Files.isDirectory(attachmentsPath)) {
            logger.info("Directory {} does not exist. Skipping Attachments posting.", attachmentsPath);
            return;
        }

        try (Stream<Path> subDirs = Files.list(attachmentsPath)) {
            subDirs
                    .filter(Files::isDirectory)
                    .forEach(subDir -> {
                        try (Stream<Path> files = Files.list(subDir)) {
                            files
                                    .filter(Files::isRegularFile)
                                    .forEach(file -> {
                                        try {
                                            byte[] fileContent = Files.readAllBytes(file);
                                            String attachmentUrl = apiEndpoint + "/serviceCatalogManagement/v4/serviceSpecification/" + newServSpec.getUuid() + "/attachment";
                                            String token = keycloakAuthenticator.getToken();

                                            String boundary = "----WebKitFormBoundaryqGHQuzNBYYgFlLIr";
                                            String formData = "--" + boundary + "\r\n" +
                                                    "Content-Disposition: form-data; name=\"afile\"; filename=\"" + file.getFileName() + "\"\r\n" +
                                                    "Content-Type: application/octet-stream\r\n\r\n" +
                                                    new String(fileContent) + "\r\n" +
                                                    "--" + boundary + "--";

                                            HttpRequest request = HttpRequest.newBuilder()
                                                    .uri(URI.create(attachmentUrl))
                                                    .header("Authorization", "Bearer " + token)
                                                    .header("Content-Type", "multipart/form-data; boundary=" + boundary)
                                                    .header("accept", "application/json, text/plain, */*")
                                                    .header("accept-encoding", "gzip, deflate, br, zstd")
                                                    .POST(HttpRequest.BodyPublishers.ofString(formData))
                                                    .build();

                                            HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

                                            if (response.statusCode() != 200) {
                                                logger.error("Failed to post attachment. HTTP status code: {}", response.statusCode());
                                                logger.error("Response body: {}", response.body());
                                            } else {
                                                logger.info("Successfully posted attachment. Response: {}", response.body());
                                            }
                                        } catch (IOException | InterruptedException e) {
                                            logger.error("Failed to process file: {} - {}", file, e.getMessage());
                                        }
                                    });
                        } catch (IOException e) {
                            logger.error("Failed to read directory: {} - {}", subDir, e.getMessage());
                        }
                    });
        } catch (IOException e) {
            logger.error("Failed to read directory: {} - {}", attachmentsPath, e.getMessage());
        }
    }

    private static void postResourceSpecificationAttachment(Path path, ResourceSpecification newResourceSpec) {
        Path basePath = Files.isDirectory(path) ? path : path.getParent();
        logger.info("BasePath for attachments: {}", basePath);
        Path attachmentsPath = basePath.resolve("attachment");
        if (!Files.exists(attachmentsPath) || !Files.isDirectory(attachmentsPath)) {
            logger.info("Directory {} does not exist. Skipping Attachments posting.", attachmentsPath);
            return;
        }

        try (Stream<Path> subDirs = Files.list(attachmentsPath)) {
            subDirs
                    .filter(Files::isDirectory)
                    .forEach(subDir -> {
                        try (Stream<Path> files = Files.list(subDir)) {
                            files
                                    .filter(Files::isRegularFile)
                                    .forEach(file -> {
                                        try {
                                            byte[] fileContent = Files.readAllBytes(file);
                                            String attachmentUrl = apiEndpoint + "/resourceCatalogManagement/v4/resourceSpecification/" + newResourceSpec.getUuid() + "/attachment";
                                            String token = keycloakAuthenticator.getToken();

                                            String boundary = "----WebKitFormBoundaryqGHQuzNBYYgFlLIr";
                                            String formData = "--" + boundary + "\r\n" +
                                                    "Content-Disposition: form-data; name=\"afile\"; filename=\"" + file.getFileName() + "\"\r\n" +
                                                    "Content-Type: application/octet-stream\r\n\r\n" +
                                                    new String(fileContent) + "\r\n" +
                                                    "--" + boundary + "--";

                                            HttpRequest request = HttpRequest.newBuilder()
                                                    .uri(URI.create(attachmentUrl))
                                                    .header("Authorization", "Bearer " + token)
                                                    .header("Content-Type", "multipart/form-data; boundary=" + boundary)
                                                    .header("accept", "application/json, text/plain, */*")
                                                    .header("accept-encoding", "gzip, deflate, br, zstd")
                                                    .POST(HttpRequest.BodyPublishers.ofString(formData))
                                                    .build();

                                            HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

                                            if (response.statusCode() != 200) {
                                                logger.error("Failed to post attachment. HTTP status code: {}", response.statusCode());
                                                logger.error("Response body: {}", response.body());
                                            } else {
                                                logger.info("Successfully posted attachment. Response: {}", response.body());
                                            }
                                        } catch (IOException | InterruptedException e) {
                                            logger.error("Failed to process file: {} - {}", file, e.getMessage());
                                        }
                                    });
                        } catch (IOException e) {
                            logger.error("Failed to read directory: {} - {}", subDir, e.getMessage());
                        }
                    });
        } catch (IOException e) {
            logger.error("Failed to read directory: {} - {}", attachmentsPath, e.getMessage());
        }
    }

    private static void postLCMRules(Path path, ServiceSpecification newServSpec) {
        Path basePath = Files.isDirectory(path) ? path : path.getParent();
        Path lcmRulesPath = basePath.resolve("serviceSpecificationLcmRules");

        if (!Files.exists(lcmRulesPath) || !Files.isDirectory(lcmRulesPath)) {
            logger.info("Directory {} does not exist. Skipping LCM Rule posting.", lcmRulesPath);
            return;
        }

        try (Stream<Path> files = Files.list(lcmRulesPath)) {
            files
                    .filter(Files::isRegularFile)
                    .filter(file -> file.toString().endsWith(".json"))
                    .forEach(file -> {
                        try {
                            LCMRuleSpecification lcmRule = objectMapper.readValue(file.toFile(), LCMRuleSpecification.class);
                            LCMRuleSpecificationCreate lcmRuleCreate = LCMRuleSpecificationCreateFromLCMRuleSpecification(lcmRule); // org.etsi.osl.tmf.lcm.model.LCMRuleSpecificationCreate.from
                            ServiceSpecificationRef serviceSpecRef = createServiceSpecificationRef(newServSpec);
                            List<ServiceSpecificationRef> serviceSpecs = new ArrayList<>();
                            serviceSpecs.add(serviceSpecRef);
                            lcmRuleCreate.setServiceSpecs(serviceSpecs);
                            String jsonContent = JsonStringReplacer.replaceJsonStrings(objectMapper.writeValueAsString(lcmRuleCreate));
                            String ruleUrl = apiEndpoint + "/lcmrulesmanagement/v1/lcmRuleSpecification";
                            String token = keycloakAuthenticator.getToken();

                            HttpResponse<String> response = sendHttpRequestWithRetry(ruleUrl, token, jsonContent, "POST", "post LCM Rule Specification", 3);

                            if (response.statusCode() != 200) {
                                logger.error("Failed to post LCM Rule Specification. HTTP status code: {}", response.statusCode());
                                logger.error("Response body: {}", response.body());
                            } else {
                                logger.info("Successfully posted LCM Rule Specification. Response: {}", response.body());
                            }
                        } catch (IOException | InterruptedException e) {
                            logger.error("Failed to process file: {} - {}", file, e.getMessage());
                        }
                    });
        } catch (IOException e) {
            logger.error("Failed to read directory: {} - {}", lcmRulesPath, e.getMessage());
        }
    }

    private static ServiceSpecificationRef createServiceSpecificationRef(ServiceSpecification serviceSpec) {
        ServiceSpecificationRef serviceSpecRef = new ServiceSpecificationRef();
        serviceSpecRef.setId(serviceSpec.getId());
        serviceSpecRef.setHref(serviceSpec.getHref());
        serviceSpecRef.setName(serviceSpec.getName());
        serviceSpecRef.setVersion(serviceSpec.getVersion());
        serviceSpecRef.setBaseType(serviceSpec.getBaseType());
        serviceSpecRef.setSchemaLocation(serviceSpec.getSchemaLocation());
        serviceSpecRef.setType(serviceSpec.getType());
        return serviceSpecRef;
    }

    // Static method to create LCMRuleSpecificationCreate from LCMRuleSpecification
    public static LCMRuleSpecificationCreate LCMRuleSpecificationCreateFromLCMRuleSpecification(LCMRuleSpecification spec) {
        LCMRuleSpecificationCreate create = new LCMRuleSpecificationCreate();
        create.setName(spec.getName());
        create.setDescription(spec.getDescription());
        create.setPriority(spec.getPriority());
        create.setLcmrulephase(spec.getLcmrulephase());
        //create.setServiceSpecs((List<ServiceSpecificationRef>) spec.getServiceSpecs());
        create.setContent(spec.getContent());
        create.setCode(spec.getCode());
        create.setLifecycleStatus(spec.getLifecycleStatus());
        create.setVersion(spec.getVersion());
        create.setValidFor(spec.getValidFor());
        create.setBaseType(spec.getBaseType());
        create.setSchemaLocation(spec.getSchemaLocation());
        create.setType(spec.getType());
        return create;
    }

    private static ServiceSpecification postServiceSpecMainProperties(ServiceSpecification servSpecFromFile) throws IOException, InterruptedException {
        ServiceSpecificationCreate newServSpec = ServiceSpecificationPayloadsMaker.createServiceSpecDataFromFile(servSpecFromFile);
        String jsonContent = JsonStringReplacer.replaceJsonStrings(objectMapper.writeValueAsString(newServSpec));
        String ruleUrl = apiEndpoint + "/serviceCatalogManagement/v4/serviceSpecification";
        String token = keycloakAuthenticator.getToken();

        HttpResponse<String> response = sendHttpRequestWithRetry(ruleUrl, token, jsonContent, "POST", "post Service Specification", 3);

        if (response.statusCode() != 200) {
            throw new IOException("Failed request with status code: " + response.statusCode());
        } else {
            ServiceSpecification newServSpecTmp = objectMapper.readValue(response.body(), ServiceSpecification.class);
            logger.info("ServiceSpecification created with UUID: {}", newServSpecTmp.getUuid());
            return newServSpecTmp;
        }
    }

    private static ResourceSpecification postResourceSpecMainProperties(ResourceSpecification resourceSpecFromFile) throws IOException, InterruptedException {
        ResourceSpecificationCreate newResourceSpec = ResourceSpecificationPayloadsMaker.createResourceSpecDataFromFile(resourceSpecFromFile);
        String jsonContent = JsonStringReplacer.replaceJsonStrings(objectMapper.writeValueAsString(newResourceSpec));
        String ruleUrl = apiEndpoint + "/resourceCatalogManagement/v4/resourceSpecification";
        String token = keycloakAuthenticator.getToken();

        HttpResponse<String> response = sendHttpRequestWithRetry(ruleUrl, token, jsonContent, "POST", "post Resource Specification", 3);

        if (response.statusCode() != 200) {
            throw new IOException("Failed request with status code: " + response.statusCode());
        } else {
            logger.info("ResourceSpecification created");
            JsonNode rootNode = objectMapper.readTree(response.body());
            String baseType = rootNode.get("@type").asText();
            if ("LogicalResourceSpecification".equals(baseType)) {
                return objectMapper.treeToValue(rootNode, LogicalResourceSpecification.class);
            } else if ("PhysicalResourceSpecification".equals(baseType)) {
                return objectMapper.treeToValue(rootNode, PhysicalResourceSpecification.class);
            } else {
                throw new IllegalArgumentException("Unsupported ResourceSpecification baseType: " + baseType);
            }
        }
    }

    private static ServiceSpecification patchServiceSpecificationCharacteristics(ServiceSpecification servSpecFromFile, ServiceSpecification servSpecToPatch) throws IOException, InterruptedException {
        ServiceSpecificationUpdate servSpecUpdate = ServiceSpecificationPayloadsMaker.updateServiceSpecDataFromFile(servSpecFromFile);
        String jsonContent = JsonStringReplacer.replaceJsonStrings(objectMapper.writeValueAsString(servSpecUpdate));
        String ruleUrl = apiEndpoint + "/serviceCatalogManagement/v4/serviceSpecification/" + servSpecToPatch.getUuid();
        String token = keycloakAuthenticator.getToken();

        HttpResponse<String> response = sendHttpRequestWithRetry(ruleUrl, token, jsonContent, "PATCH", "patch Service Specification", 3);

        if (response.statusCode() != 200) {
            throw new IOException("Failed request with status code: " + response.statusCode());
        } else {
            return objectMapper.readValue(response.body(), ServiceSpecification.class);
        }
    }

    private static ResourceSpecification patchResourceSpecificationCharacteristics(ResourceSpecification resourceSpecFromFile, ResourceSpecification resourceSpecToPatch) throws IOException, InterruptedException {
        ResourceSpecificationUpdate resourceSpecUpdate = ResourceSpecificationPayloadsMaker.updateResourceSpecDataFromFile(resourceSpecFromFile);
        String jsonContent = JsonStringReplacer.replaceJsonStrings(objectMapper.writeValueAsString(resourceSpecUpdate));
        String ruleUrl = apiEndpoint + "/resourceCatalogManagement/v4/resourceSpecification/" + resourceSpecToPatch.getUuid();
        String token = keycloakAuthenticator.getToken();

        HttpResponse<String> response = sendHttpRequestWithRetry(ruleUrl, token, jsonContent, "PATCH", "patch Service Specification", 3);

        if (response.statusCode() != 200) {
            throw new IOException("Failed request with status code: " + response.statusCode());
        } else {
            logger.info("response.body():"+response.body());
            JsonNode rootNode = objectMapper.readTree(response.body());
            String baseType = rootNode.get("@type").asText();
            if ("LogicalResourceSpecification".equals(baseType)) {
                return objectMapper.treeToValue(rootNode, LogicalResourceSpecification.class);
            } else if ("PhysicalResourceSpecification".equals(baseType)) {
                return objectMapper.treeToValue(rootNode, PhysicalResourceSpecification.class);
            } else {
                throw new IllegalArgumentException("Unsupported ResourceSpecification baseType: " + baseType);
            }
        }
    }


    private static void postResourceSpec(Path path, boolean isPhysical) {
        try {
            ResourceSpecification resourceSpecFromFile;
            if (isPhysical) {
                resourceSpecFromFile = objectMapper.readValue(path.toFile(), PhysicalResourceSpecification.class);
            } else {
                resourceSpecFromFile = objectMapper.readValue(path.toFile(), LogicalResourceSpecification.class);
            }

            // If the ServiceSpecification name and version already exists in the database, skip it to avoid duplicates
            if (uuid2nameversion.containsValue(resourceSpecFromFile.getName() + " " + resourceSpecFromFile.getVersion())) {
                logger.info("{} with name {} and version {} already exists. Skipping.",
                        isPhysical ? "PhysicalResourceSpecification" : "LogicalResourceSpecification",
                        resourceSpecFromFile.getName(), resourceSpecFromFile.getVersion());
                return;
            }

            List<ResourceSpecification> resourceSpecificationsInOpenslice = getResourceSpecificationsInOpenslice();
            for (ResourceSpecification resourceSpecificationInOpenSlice : resourceSpecificationsInOpenslice) {
                if (resourceSpecificationInOpenSlice.getName().equals(resourceSpecFromFile.getName()) &&
                        resourceSpecificationInOpenSlice.getVersion().equals(resourceSpecFromFile.getVersion())) {
                    uuidmap.put(resourceSpecFromFile.getUuid(), resourceSpecificationInOpenSlice.getUuid());
                    idmap.put(resourceSpecFromFile.getId(), resourceSpecificationInOpenSlice.getId());
                    uuid2nameversion.put(resourceSpecificationInOpenSlice.getUuid(),
                            resourceSpecificationInOpenSlice.getName() + " " + resourceSpecificationInOpenSlice.getVersion());
                    logger.info("ResourceSpecification with name {} and version {} already exists. Skipping.",
                            resourceSpecFromFile.getName(), resourceSpecFromFile.getVersion());
                    return;
                }
            }

            ResourceSpecification newResSpec = postResourceSpecMainProperties(resourceSpecFromFile);
            uuidmap.put(resourceSpecFromFile.getUuid(), newResSpec.getUuid());
            idmap.put(resourceSpecFromFile.getId(), newResSpec.getId());
            uuid2nameversion.put(newResSpec.getUuid(), newResSpec.getName() + " " + newResSpec.getVersion());
            newResSpec = patchResourceSpecificationCharacteristics(resourceSpecFromFile, newResSpec);
            postResourceSpecificationAttachment(path, newResSpec);
            newResSpec = patchResourceSpecificationResourceRelationships(resourceSpecFromFile, newResSpec);
        } catch (IOException | InterruptedException e) {
            logger.error("Process halted due to failure: {}", e.getMessage());
        }
    }

    private static void postPhysicalResourceSpec(Path path) {
        postResourceSpec(path, true);
    }

    private static void postLogicalResourceSpec(Path path) {
        postResourceSpec(path, false);
    }

}
