package org.etsi.osl.util;

import com.fasterxml.jackson.databind.*;
import com.fasterxml.jackson.databind.node.*;
import org.yaml.snakeyaml.Yaml;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.util.*;

public class JsonStringReplacer {

    // Logger instance
    private static final Logger logger = LoggerFactory.getLogger(JsonStringReplacer.class);

    // Static variable to hold mappings
    private static Map<String, String> mappings = new HashMap<>();

    /**
     * Sets the YAML file path and loads mappings.
     * If the file is missing, it sets mappings to an empty map and continues.
     *
     * @param yamlFilePath The path to the YAML mappings file.
     */
    public static void setYamlFilePath(String yamlFilePath) {
        try {
            mappings = loadYamlMappings(yamlFilePath);
        } catch (FileNotFoundException e) {
            // Mappings file is missing; proceed with empty mappings
            logger.warn("Mappings file not found: {}. Proceeding without replacements.", yamlFilePath);
            mappings = new HashMap<>(); // Ensure mappings is empty
        } catch (IOException e) {
            // Other I/O errors
            logger.error("Error reading mappings file: {}. Proceeding without replacements.", e.getMessage());
            mappings = new HashMap<>(); // Ensure mappings is empty
        } catch (Exception e) {
            // Handle any other exceptions
            logger.error("Unexpected error: {}. Proceeding without replacements.", e.getMessage(), e);
            mappings = new HashMap<>(); // Ensure mappings is empty
        }
    }

    /**
     * Replaces strings in a JSON content based on the loaded mappings.
     * If mappings are empty, it returns the original JSON content.
     *
     * @param jsonContent The input JSON content as a string.
     * @return The modified JSON content as a string.
     * @throws IOException If an I/O error occurs during processing.
     */
    public static String replaceJsonStrings(String jsonContent) throws IOException {
        // If mappings are empty, return the original JSON content
        if (mappings == null || mappings.isEmpty()) {
            logger.info("Mappings are empty. Returning original JSON content.");
            return jsonContent;
        }

        // Create an ObjectMapper for JSON parsing
        ObjectMapper mapper = new ObjectMapper();

        // Parse the JSON content into a JsonNode
        JsonNode rootNode = mapper.readTree(jsonContent);

        // Perform string replacements
        replaceStrings(rootNode);

        // Convert the modified JsonNode back into a JSON string
        String result = mapper.writerWithDefaultPrettyPrinter().writeValueAsString(rootNode);
        logger.debug("Replacements completed successfully.");
        return result;
    }

    /**
     * Loads mappings from a YAML file.
     *
     * @param yamlFilePath The file path to the YAML mappings file.
     * @return A Map containing the mappings.
     * @throws IOException If an error occurs while reading the YAML file.
     */
    private static Map<String, String> loadYamlMappings(String yamlFilePath) throws IOException {
        Yaml yaml = new Yaml();
        Map<String, String> loadedMappings = new HashMap<>();

        try (InputStream inputStream = new FileInputStream(yamlFilePath)) {
            Map<String, Object> yamlMaps = yaml.load(inputStream);

            if (yamlMaps == null || !yamlMaps.containsKey("mappings")) {
                logger.warn("The YAML file does not contain a 'mappings' key. Proceeding without replacements.");
                return loadedMappings; // Return empty mappings
            }

            Object mappingsObj = yamlMaps.get("mappings");

            if (mappingsObj instanceof List) {
                List<?> mappingsList = (List<?>) mappingsObj;
                for (Object item : mappingsList) {
                    if (item instanceof Map) {
                        Map<?, ?> mapItem = (Map<?, ?>) item;
                        Object fromObj = mapItem.get("from");
                        Object toObj = mapItem.get("to");

                        if (fromObj instanceof String && toObj instanceof String) {
                            loadedMappings.put((String) fromObj, (String) toObj);
                        } else {
                            logger.warn("Invalid mapping entry: 'from' and 'to' must be strings. Skipping entry.");
                        }
                    } else {
                        logger.warn("Invalid mapping format. Each mapping entry must be a map with 'from' and 'to' keys. Skipping entry.");
                    }
                }
            } else if (mappingsObj instanceof Map) {
                Map<?, ?> mappingsMap = (Map<?, ?>) mappingsObj;
                for (Map.Entry<?, ?> entry : mappingsMap.entrySet()) {
                    if (entry.getKey() instanceof String && entry.getValue() instanceof String) {
                        loadedMappings.put((String) entry.getKey(), (String) entry.getValue());
                    } else {
                        logger.warn("Invalid mapping entry: keys and values must be strings. Skipping entry.");
                    }
                }
            } else {
                logger.warn("Unrecognized 'mappings' format in YAML file. Proceeding without replacements.");
            }
        }

        logger.info("Loaded {} mappings from YAML file.", loadedMappings.size());
        return loadedMappings;
    }

    /**
     * Recursively replaces strings in a JsonNode based on the loaded mappings.
     *
     * @param node The JsonNode to process.
     */
    private static void replaceStrings(JsonNode node) {
        if (node == null || mappings == null || mappings.isEmpty()) {
            return;
        }

        if (node.isObject()) {
            ObjectNode objNode = (ObjectNode) node;
            Iterator<Map.Entry<String, JsonNode>> fields = objNode.fields();
            while (fields.hasNext()) {
                Map.Entry<String, JsonNode> entry = fields.next();
                JsonNode childNode = entry.getValue();

                if (childNode.isTextual()) {
                    String textValue = childNode.asText();
                    if (mappings.containsKey(textValue)) {
                        String replacement = mappings.get(textValue);
                        objNode.put(entry.getKey(), replacement);
                        logger.debug("Replaced text '{}' with '{}' in object field '{}'.", textValue, replacement, entry.getKey());
                    }
                } else {
                    replaceStrings(childNode);
                }
            }
        } else if (node.isArray()) {
            ArrayNode arrayNode = (ArrayNode) node;
            for (int i = 0; i < arrayNode.size(); i++) {
                JsonNode element = arrayNode.get(i);

                if (element.isTextual()) {
                    String textValue = element.asText();
                    if (mappings.containsKey(textValue)) {
                        String replacement = mappings.get(textValue);
                        arrayNode.set(i, new TextNode(replacement));
                        logger.debug("Replaced text '{}' with '{}' in array index {}.", textValue, replacement, i);
                    }
                } else {
                    replaceStrings(element);
                }
            }
        }
        // Other node types are not modified
    }

    // Example usage in the main method
    public static void main(String[] args) {
        String jsonInput = "{ \"greeting\": \"string1\", \"farewell\": \"string2\", \"nested\": { \"message\": \"string3\" }, \"array\": [\"string1\", \"hello\", \"string2\"] }";
        String yamlFilePath = "org/etsi/osl/util/mappings.yaml"; // Adjust the path as needed

        // Set the YAML path from another class or context
        JsonStringReplacer.setYamlFilePath(yamlFilePath);

        try {
            // Perform the replacement
            String result = JsonStringReplacer.replaceJsonStrings(jsonInput);

            // Output the modified JSON
            System.out.println("Modified JSON:");
            System.out.println(result);
        } catch (IOException e) {
            logger.error("An error occurred during JSON processing: {}", e.getMessage(), e);
        }
    }
}
