package org.osl.etsi.util;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.file.Files;
import java.util.*;

import org.etsi.osl.tmf.common.model.AttachmentRef;
import org.etsi.osl.tmf.common.model.AttachmentRefOrValue;
import org.etsi.osl.tmf.common.model.service.ServiceSpecificationRef;
import org.etsi.osl.tmf.lcm.model.LCMRuleSpecification;
import org.etsi.osl.tmf.rcm634.model.LogicalResourceSpecification;
import org.etsi.osl.tmf.rcm634.model.PhysicalResourceSpecification;
import org.etsi.osl.tmf.rcm634.model.ResourceSpecification;
import org.etsi.osl.tmf.rcm634.model.ResourceSpecificationRef;
import org.etsi.osl.tmf.rcm634.model.ResourceSpecificationRelationship;
import org.etsi.osl.tmf.scm633.model.ServiceSpecRelationship;
import org.etsi.osl.tmf.scm633.model.ServiceSpecification;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServiceSpecificationFetcher {
    private static final Logger logger = LoggerFactory.getLogger(ServiceSpecificationFetcher.class);
    private static final HttpClient client = HttpClient.newHttpClient();
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private static String apiEndpoint=null;
    private static KeycloakAuthenticator keycloakAuthenticator = null;
    private static final Properties config = new Properties();
    private static String configFilePath = null;

    public static void main(String[] args) throws IOException {
        String serviceSpecUUid = null;
        // Parse command-line arguments
        for (int i = 0; i < args.length; i++) {
            switch (args[i]) {
                case "--configfile":
                    if (i + 1 < args.length) {
                        configFilePath = args[++i];
                        logger.info("Config file path set to: " + configFilePath);
                        try (InputStream input = new FileInputStream(configFilePath)) {
                            config.load(input);
                            apiEndpoint = config.getProperty("sourceApiEndpoint.url");
                            logger.info("Configuration loaded successfully from " + configFilePath);
                        } catch (IOException ex) {
                            logger.error("Failed to load configuration from " + configFilePath, ex);
                            throw ex;
                        }
                        keycloakAuthenticator= new KeycloakAuthenticator(config);
                    } else {
                        logger.error("Missing value for --configfile");
                        printUsageAndExit();
                    }
                    break;
                case "--servicespecuuid":
                    if (i + 1 < args.length) {
                        serviceSpecUUid = args[++i];
                        logger.info("Service spec uuid set to: " + serviceSpecUUid);
                    } else {
                        logger.error("Missing value for --servicespecuuid");
                        printUsageAndExit();
                    }
                    break;
                default:
                    logger.error("Unknown argument: " + args[i]);
                    printUsageAndExit();
            }
        }

        // Validate required arguments
        if (configFilePath == null || serviceSpecUUid == null) {
            logger.error("Missing required arguments.");
            printUsageAndExit();
        }

        // Continue with the rest of your application logic
        logger.info("Application started successfully with provided configurations.");
        try {
            fetchServiceSpecification(serviceSpecUUid, null);
            logger.info("All data has been fetched and saved hierarchically.");
        } catch (IOException | InterruptedException e) {
            e.printStackTrace();
        }
    }

    /**
     * Prints usage instructions and exits the application.
     */
    private static void printUsageAndExit() {
        String usage = "Usage:\n" +
                "  java -jar servicespecificationfetcher.jar --configfile /path/to/config.yaml --servicespecuuid servicespecuuid\n\n" +
                "Parameters:\n" +
                "  --configfile         Path to the configuration file.\n" +
                "  --servicespecuuid    Path to the service specification uuid.";
        logger.info(usage);
        System.exit(1);
    }

    private static void loadConfig() {
        try (InputStream input = ServiceSpecificationFetcher.class.getClassLoader().getResourceAsStream("config.properties")) {
            if (input == null) {
                throw new IOException("Unable to find config.properties");
            }
            config.load(input);
        } catch (IOException ex) {
            ex.printStackTrace();
        }
    }

    private static ServiceSpecification fetchServiceSpecification(String uuid, File parentDirectory) throws IOException, InterruptedException {
        String url = apiEndpoint + "/serviceCatalogManagement/v4/serviceSpecification/" + uuid;
        HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(url))
                .GET()
                .build();
        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() != 200) {
            throw new IOException("Failed to get a valid response for "+url+". HTTP status code: " + response.statusCode());
        }

        if (response.body() == null || response.body().isEmpty()) {
            logger.error("No content to map for URL: " + url);
            return null;
        }

        ServiceSpecification serviceSpecification = objectMapper.readValue(response.body(), ServiceSpecification.class);
        if (serviceSpecification != null && serviceSpecification.getId() != null) {
            logger.info("ServiceSpecification found. Creating folder " + serviceSpecification.getId());
            File serviceSpecificationFolder = new File(parentDirectory, serviceSpecification.getId());
            if (!serviceSpecificationFolder.exists()) {
                serviceSpecificationFolder.mkdirs();
            }
            saveJsonToFile(serviceSpecification, serviceSpecificationFolder, serviceSpecification.getId() + ".json");
            fetchAndSaveAttachments(serviceSpecification, serviceSpecificationFolder);
            fetchLCMRuleSpecification(uuid, serviceSpecificationFolder);

            File relationshipDir = new File(serviceSpecificationFolder, "serviceSpecificationServiceRelationships");
            if (!relationshipDir.exists()) {
                relationshipDir.mkdirs();
            }

            if (serviceSpecification.getServiceSpecRelationship() != null) {
                for (ServiceSpecRelationship serviceRelationship : serviceSpecification.getServiceSpecRelationship()) {
                    fetchServiceSpecification(serviceRelationship.getId(), relationshipDir);
                }
            }

            logger.info("Creating serviceSpecificationResourceRelationships folder for "+serviceSpecification.getId());

            File resourceRelationshipDir = new File(serviceSpecificationFolder, "serviceSpecificationResourceRelationships");
            if (!resourceRelationshipDir.exists()) {
                resourceRelationshipDir.mkdirs();
            }

            if (serviceSpecification.getResourceSpecification()!= null) {
                logger.info("Resource Relationships for "+serviceSpecification.getId()+" found!");

                for (ResourceSpecificationRef resourceRelationship : serviceSpecification.getResourceSpecification()) {
                    String relatedUrl = apiEndpoint + "/resourceCatalogManagement/v4/resourceSpecification/" + resourceRelationship.getId(); // Now using getId()
                    logger.info("Fetching from "+relatedUrl);
                    fetchResourceSpecification(relatedUrl, resourceRelationshipDir);
                }
            }
            else
            {
                logger.info("Resource Relationships for "+serviceSpecification.getId()+" NOT found!");

            }
        }
        return serviceSpecification;
    }

    private static void fetchLCMRuleSpecification(String serviceSpecId, File parentDirectory) throws IOException, InterruptedException {
        String url = apiEndpoint + "/lcmrulesmanagement/v1/lcmRuleSpecification/serviceSpec/" + serviceSpecId;
        String token = keycloakAuthenticator.getToken();
        HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(url))
                .header("Authorization", "Bearer " + token)
                .GET()
                .build();
        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() != 200) {
            throw new IOException("Failed to get a valid response for "+url+". HTTP status code: " + response.statusCode());
        }

        File lcmRulesDirectory = new File(parentDirectory, "serviceSpecificationLcmRules");
        if (!lcmRulesDirectory.exists()) {
            lcmRulesDirectory.mkdirs();
        }

        try {
            List<String> ruleIds = parseJsonForIds(response.body());
            if (ruleIds != null && !ruleIds.isEmpty()) {
                // Save a list of rule IDs to a file, assuming this is still required
                //saveJsonToFile(ruleIds, new File(parentDirectory, serviceSpecId + "-lcmRuleIds.json"));
                saveJsonToFile(response.body(), new File(parentDirectory, serviceSpecId + "-lcmRuleIds.json"));

                // Iterate over each ID to fetch and save detailed rule information
                for (String ruleId : ruleIds) {
                    fetchAndSaveDetailedRule(ruleId, lcmRulesDirectory);
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private static void fetchAndSaveDetailedRule(String ruleId, File parentDirectory) throws IOException, InterruptedException {
        String ruleUrl = apiEndpoint + "/lcmrulesmanagement/v1/lcmRuleSpecification/" + ruleId;
        String token = keycloakAuthenticator.getToken();
        HttpRequest detailedRequest = HttpRequest.newBuilder()
                .uri(URI.create(ruleUrl))
                .header("Authorization", "Bearer " + token)
                .GET()
                .build();
        HttpResponse<String> detailedResponse = client.send(detailedRequest, HttpResponse.BodyHandlers.ofString());

        if (detailedResponse.statusCode() != 200) {
            logger.error("Failed to get a valid response for rule ID: " + ruleId + " HTTP status code: " + detailedResponse.statusCode());
            return;
        }
        LCMRuleSpecification detailedRule = objectMapper.readValue(detailedResponse.body(), LCMRuleSpecification.class);
        detailedRule.setServiceSpecs(new HashSet<ServiceSpecificationRef>());
        File ruleFile = new File(parentDirectory, ruleId + ".json");
        saveJsonToFile(detailedRule, ruleFile);
    }

    private static ResourceSpecification fetchResourceSpecification(String url, File parentDirectory) throws IOException, InterruptedException {
        HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(url))
                .GET()
                .build();
        HttpResponse<String> response = client.send(request, HttpResponse.BodyHandlers.ofString());

        if (response.statusCode() != 200) {
            throw new IOException("Failed to get a valid response. HTTP status code: " + response.statusCode());
        }

        if (response.body() == null || response.body().isEmpty()) {
            logger.error("No content to map for URL: " + url);
            return null;
        }

        ResourceSpecification resSpec = null;
        try {
            JsonNode rootNode = objectMapper.readTree(response.body());
            JsonNode typeNode = rootNode.get("@type");

            if (typeNode != null) {
                String type = typeNode.asText();
                ObjectMapper objectMapper = new ObjectMapper();
                if ("LogicalResourceSpecification".equals(type)) {
                    resSpec = objectMapper.readValue(response.body(), LogicalResourceSpecification.class);
                    logger.info("The JSON contains a LogicalResourceSpecification.");
                } else if ("PhysicalResourceSpecification".equals(type)) {
                    resSpec = objectMapper.readValue(response.body(), PhysicalResourceSpecification.class);
                    logger.info("The JSON contains a PhysicalResourceSpecification.");
                } else {
                    logger.info("The JSON contains an unknown type: " + type);
                }
            } else {
                logger.info("The JSON does not contain a @type field.");
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        if (resSpec != null && resSpec.getId() != null) {
            logger.info("Creating "+resSpec.getId()+" folder!");
            //logger.info("Creating "+resSpec.getName()+" folder!");
            File resourceSpecificationFolder = new File(parentDirectory, resSpec.getId());
            //File resourceSpecificationFolder = new File(parentDirectory, resSpec.getName());
            if (!resourceSpecificationFolder.exists()) {
                resourceSpecificationFolder.mkdirs();
            }
            saveResourceSpecificationJsonToFile(resSpec, resourceSpecificationFolder, resSpec.getId() + ".json");
            //saveResourceSpecificationJsonToFile(resSpec, resourceSpecificationFolder, resSpec.getName() + ".json");
            fetchAndSaveAttachments(resSpec, resourceSpecificationFolder);

            File relationshipDir = new File(resourceSpecificationFolder, "resourceSpecificationResourceRelationships");
            if (!relationshipDir.exists()) {
                relationshipDir.mkdirs();
            }

            if (resSpec.getResourceSpecRelationship()!= null) {
                logger.info("Resource Relationships for "+resSpec.getId()+" found!");
                //logger.info("Resource Relationships for "+resSpec.getName()+" found!");
                for (ResourceSpecificationRelationship relationship : resSpec.getResourceSpecRelationship()) {
                    String relatedUrl = apiEndpoint + "/resourceCatalogManagement/v4/resourceSpecification/" + relationship.getId(); // Now using getId()
                    fetchResourceSpecification(relatedUrl, relationshipDir);
                }
            }
            else
            {
                logger.info("Resource Relationships for "+resSpec.getId()+" NOT found!");

            }
        }
        else
        {
            logger.info("resSpec != null && resSpec.getId() != null");
        }
        return resSpec;
    }

    // Method for ServiceSpecifications
    private static void fetchAndSaveAttachments(ServiceSpecification spec, File parentFolder) throws IOException, InterruptedException {
        if (spec.getAttachment() != null) {
            for (AttachmentRef attachment : spec.getAttachment()) {
                String attachmentUrl = attachment.getUrl();
                HttpRequest attachmentRequest = HttpRequest.newBuilder()
                        .uri(URI.create(apiEndpoint + attachmentUrl)) // Assuming the base URL needs to be prefixed
                        .GET()
                        .build();
                HttpResponse<byte[]> attachmentResponse = client.send(attachmentRequest, HttpResponse.BodyHandlers.ofByteArray());

                if (attachmentResponse.statusCode() == 200) {
                    String relativePath = extractRelativePath(attachmentUrl);
                    File attachmentDir = new File(parentFolder, relativePath.substring(0, relativePath.lastIndexOf('/')));
                    if (!attachmentDir.exists()) {
                        attachmentDir.mkdirs(); // Make sure the directory structure exists
                    }
                    File attachmentFile = new File(attachmentDir, relativePath.substring(relativePath.lastIndexOf('/') + 1));
                    Files.write(attachmentFile.toPath(), attachmentResponse.body());
                    logger.info("Attachment saved to " + attachmentFile.getPath());
                } else {
                    logger.info("Failed to fetch attachment at " + attachmentUrl);
                }
            }
        }
    }

    // Overloaded method for ResourceSpecifications
    private static void fetchAndSaveAttachments(ResourceSpecification spec, File parentFolder) throws IOException, InterruptedException {
        if (spec.getAttachment() != null) {
            for (AttachmentRefOrValue attachment : spec.getAttachment()) {
                String attachmentUrl = attachment.getUrl();
                HttpRequest attachmentRequest = HttpRequest.newBuilder()
                        .uri(URI.create(apiEndpoint + attachmentUrl)) // Assuming the base URL needs to be prefixed
                        .GET()
                        .build();
                HttpResponse<byte[]> attachmentResponse = client.send(attachmentRequest, HttpResponse.BodyHandlers.ofByteArray());

                if (attachmentResponse.statusCode() == 200) {
                    String relativePath = extractRelativePath(attachmentUrl);
                    File attachmentDir = new File(parentFolder, relativePath.substring(0, relativePath.lastIndexOf('/')));
                    if (!attachmentDir.exists()) {
                        attachmentDir.mkdirs(); // Make sure the directory structure exists
                    }
                    File attachmentFile = new File(attachmentDir, relativePath.substring(relativePath.lastIndexOf('/') + 1));
                    Files.write(attachmentFile.toPath(), attachmentResponse.body());
                    logger.info("Attachment saved to " + attachmentFile.getPath());
                } else {
                    logger.info("Failed to fetch attachment at " + attachmentUrl);
                }
            }
        }
    }

    public static List<String> parseJsonForIds(String json) throws IOException {
        // Parse the JSON string into a JsonNode
        JsonNode rootNode = objectMapper.readTree(json);
        List<String> ids = new ArrayList<>();

        // Check if the root node is an array and iterate over it
        if (rootNode.isArray()) {
            for (JsonNode node : rootNode) {
                // Extract the id from each object in the array
                String id = node.get("id").asText();
                ids.add(id);
            }
        }

        return ids;
    }

    private static String extractRelativePath(String url) {
        URI uri = URI.create(url);
        String path = uri.getPath();
        return path.substring(path.indexOf("/attachment/") + 1); // Extract the path after "/serviceSpecification/"
    }

    private static String extractFilename(String url) {
        URI uri = URI.create(url);
        String path = uri.getPath();
        return path.substring(path.lastIndexOf('/') + 1);
    }

    private static void saveJsonToFile(ServiceSpecification spec, File parentFolder, String filename) throws IOException {
        File file = new File(parentFolder, filename);
        objectMapper.writerWithDefaultPrettyPrinter().writeValue(file, spec);
        logger.info("Saved JSON to " + file.getPath());
    }

    private static void saveJsonToFile(Object data, File file) throws IOException {
        objectMapper.writerWithDefaultPrettyPrinter().writeValue(file, data);
        logger.info("Saved JSON to " + file.getPath());
    }

    private static void saveResourceSpecificationJsonToFile(ResourceSpecification spec, File parentFolder, String filename) throws IOException {
        File file = new File(parentFolder, filename);
        objectMapper.writerWithDefaultPrettyPrinter().writeValue(file, spec);
        logger.info("Saved JSON to " + file.getPath());
    }
}
