import { dirname, join } from "path";
import { writeFileSync, existsSync, mkdirSync, readFileSync } from "fs";
import { fileURLToPath } from "url";

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);
const routesFilePathResources = join(__dirname, "../src/app/app-resources-routing.module.ts");
const routesFilePathTesting = join(__dirname, "../src/app/app-testing-routing.module.ts");
const routesFilePathServices = join(__dirname, "../src/app/app-services-routing.module.ts");
const routesFilePathProducts = join(__dirname, "../src/app/app-products-routing.module.ts");

//read route module files as plaintext and extract paths
function extractRoutesFromFile(filePath: string, parent: string): string[] {
  try {
    const fileContent = readFileSync(filePath, "utf-8");

    // Regex to match route paths (excluding those with `/:id`)
    const pathRegex = /path:\s*['"]([^'":]+)['"]/g;
    let match;
    const paths: string[] = [];

    while ((match = pathRegex.exec(fileContent)) !== null) {
      if (!match[1].includes("/:id")) {
        paths.push(`${parent}/${match[1]}`);
      }
    }

    return paths;
  } catch (error) {
    console.error(`❌ Error reading file: ${filePath}`, error);
    return [];
  }
}

// Extract paths from every file
let resourceRoutesStr = extractRoutesFromFile(routesFilePathResources, "resources");
let servicesRoutesStr = extractRoutesFromFile(routesFilePathServices, "services");
let testingRoutesStr = extractRoutesFromFile(routesFilePathTesting, "testing");
let productRoutesStr = extractRoutesFromFile(routesFilePathProducts, "products");

let allRoutes = resourceRoutesStr.concat(servicesRoutesStr, testingRoutesStr, productRoutesStr);
allRoutes.push('/products/individual_update/myuser', '/testing/individual_update/myuser', '/services/individual_update/myuser', '/resources/individual_update/myuser');
let protectedRoutes: string[] = allRoutes.filter((el) => { return el!='services/services_marketplace' && el!='products/marketplace' 
    && el!='resources/resources_marketplace'})


// Ensure `cypress/fixtures` directory exists
const fixturesDir = join(__dirname, "../cypress/fixtures");
if (!existsSync(fixturesDir)) {
  mkdirSync(fixturesDir, { recursive: true });
}

// Convert extracted routes to JSON and save to file
const jsonOutput = JSON.stringify({ routes: protectedRoutes }, null, 2);
const outputPath = join(fixturesDir, "routes.json");
writeFileSync(outputPath, jsonOutput, "utf-8");

console.log("✅ Routes JSON file generated successfully!");
