package org.etsi.osl.tmf.gsm674.api;

import io.swagger.v3.oas.annotations.Parameter;
import jakarta.validation.Valid;
import org.etsi.osl.model.nfv.UserRoleType;
import org.etsi.osl.tmf.gsm674.model.GeographicSite;
import org.etsi.osl.tmf.gsm674.reposervices.GeographicSiteManagementService;
import org.etsi.osl.tmf.pm632.model.Individual;
import org.etsi.osl.tmf.pm632.reposervices.IndividualRepoService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;

import java.security.Principal;
import java.util.List;

@Controller
@RequestMapping("/geographicSiteManagement/v5/")
public class GeographicSiteManagementApiController implements GeographicSiteManagementApi{

    private static final String COULD_NOT_SERIALIZE="Couldn't serialize response for content type application/json";
    private final GeographicSiteManagementService geographicSiteManagementService;

    private final IndividualRepoService individualRepoService;

    @Autowired
    public GeographicSiteManagementApiController(GeographicSiteManagementService geographicSiteManagementService, IndividualRepoService individualRepoService) {
        this.geographicSiteManagementService = geographicSiteManagementService;
        this.individualRepoService = individualRepoService;
    }

    @PreAuthorize("hasAnyAuthority('ROLE_USER')" )
    @Override
    public ResponseEntity<List<GeographicSite>> listGeographicSite() {


        try {
            return new ResponseEntity<>(geographicSiteManagementService.findAllGeographicSites(), HttpStatus.OK);

        } catch (Exception e) {
            log.error(COULD_NOT_SERIALIZE, e);
            return new ResponseEntity<>(HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }

    @PreAuthorize("hasAnyAuthority('ROLE_USER', 'ROLE_ADMIN')")
    @Override
    public ResponseEntity<GeographicSite> retrieveGeographicSite(Principal principal, @PathVariable("id") String id) {


        try {
            Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
            if ( id.equals( "myuser" ) ) {

                log.debug("principal=  " + principal.toString());

                Individual ind = individualRepoService.findByUsername(principal.getName());

                GeographicSite gs= geographicSiteManagementService.findGeographicSiteByRelatedPartyId(ind.getId());
                if (gs==null) gs =new GeographicSite();
                return new ResponseEntity<GeographicSite>(gs,HttpStatus.OK);

            } else if ( authentication.getAuthorities().contains( new SimpleGrantedAuthority( UserRoleType.ROLE_ADMIN.getValue()  ) ) ){
                return new ResponseEntity<GeographicSite>(geographicSiteManagementService.findGeographicSiteByUUID(id), HttpStatus.OK);

            }else {
                return new ResponseEntity< GeographicSite >(HttpStatus.FORBIDDEN );
            }

        } catch (Exception e) {
            log.error(COULD_NOT_SERIALIZE, e);
            return new ResponseEntity<>(HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }

    @PreAuthorize("hasAnyAuthority('ROLE_USER')" )
    @Override
    public ResponseEntity<GeographicSite> createGeographicSite(
            @Parameter(description = "The geographic site to be created", required = true) @Valid @RequestBody GeographicSite geographicSite
    ) {

        try {

            GeographicSite c = geographicSiteManagementService.createGeographicSite(geographicSite);

            return new ResponseEntity<>(c, HttpStatus.OK);


        } catch (Exception e) {
            log.error(COULD_NOT_SERIALIZE, e);
            return new ResponseEntity<>(HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }

    @PreAuthorize("hasAnyAuthority('ROLE_USER')" )
    @Override
    public ResponseEntity<Void> deleteGeographicSite(
            @Parameter(description = "Identifier of the geographic site", required = true) @PathVariable("id") String id) {

        try {
            return new ResponseEntity<>(geographicSiteManagementService.deleteGeographicSiteById(id), HttpStatus.OK);
        }catch (Exception e) {
            log.error(COULD_NOT_SERIALIZE, e);
            return new ResponseEntity<>(HttpStatus.INTERNAL_SERVER_ERROR);
        }

    }


    @PreAuthorize("hasAnyAuthority('ROLE_USER')" )
    @Override
    public ResponseEntity<GeographicSite> patchGeographicalSite(Principal principal,
            @Parameter(description = "Identifier of the ServiceOrder", required = true) @PathVariable("id") String id,
            @Parameter(description = "The ServiceOrder to be updated", required = true) @Valid @RequestBody GeographicSite geographicSite) {
        try{
            GeographicSite c;
                Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
                if ( id.equals( "myuser" ) ) {

                    log.debug("principal=  " + principal.toString());

                    Individual ind = individualRepoService.findByUsername(principal.getName());
                    GeographicSite gs = geographicSiteManagementService.findGeographicSiteByRelatedPartyId(ind.getId());
                     c = geographicSiteManagementService.updateGeographicSite(gs.getUuid(), geographicSite);
                }else{
                    c = geographicSiteManagementService.updateGeographicSite(id, geographicSite);

                }
        return new ResponseEntity<>(c, HttpStatus.OK);
        }catch (Exception e){
            log.error(COULD_NOT_SERIALIZE, e);
            return new ResponseEntity<>(HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }
}
