/*-
 * ========================LICENSE_START=================================
 * org.etsi.osl.tmf.api
 * %%
 * Copyright (C) 2019 - 2021 openslice.io
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =========================LICENSE_END==================================
 */
package org.etsi.osl.tmf.pcm620.reposervices;

import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.hibernate5.jakarta.Hibernate5JakartaModule;
import org.etsi.osl.tmf.common.model.ELifecycle;
import org.etsi.osl.tmf.common.model.TimePeriod;
import org.etsi.osl.tmf.pcm620.model.Catalog;
import org.etsi.osl.tmf.pcm620.model.CatalogCreate;
import org.etsi.osl.tmf.pcm620.model.CatalogUpdate;
import org.etsi.osl.tmf.pcm620.model.Category;
import org.etsi.osl.tmf.pcm620.model.CategoryRef;
import org.etsi.osl.tmf.pcm620.repo.ProductCatalogRepository;
import org.etsi.osl.tmf.scm633.model.ServiceCatalog;
import org.etsi.osl.tmf.scm633.model.ServiceCategory;
import org.etsi.osl.tmf.scm633.model.ServiceCategoryRef;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import jakarta.validation.Valid;

@Service
@Transactional
public class ProductCatalogRepoService {


	@Autowired
	ProductCatalogRepository catalogRepo;

	@Autowired
	ProductCategoryRepoService categRepoService;

	@Autowired
	CatalogNotificationService catalogNotificationService;
	
	
	public Catalog addCatalog(Catalog c) {
		Catalog savedCatalog = this.catalogRepo.save(c);
		
		// Publish catalog create notification
		catalogNotificationService.publishCatalogCreateNotification(savedCatalog);
		
		return savedCatalog;
	}

	public Catalog addCatalog(@Valid CatalogCreate serviceCat) {

		Catalog sc = new Catalog();

		sc = updateCatalogDataFromAPICall(sc, serviceCat);
		Catalog savedCatalog = this.catalogRepo.save(sc);
		
		// Publish catalog create notification
		catalogNotificationService.publishCatalogCreateNotification(savedCatalog);
		
		return savedCatalog;
	}
	
	public List<Catalog> findAll() {
		return (List<Catalog>) this.catalogRepo.findByOrderByName();
	}
	
	public Catalog findById(String id) {
		Optional<Catalog> optionalCat = this.catalogRepo.findByUuid(id);
		return optionalCat.orElse(null);
	}
	

	public Catalog findByName(String aName) {
		Optional<Catalog> optionalCat = this.catalogRepo.findByName( aName );
		return optionalCat.orElse(null);
	}

	public Void deleteById(String id) {
		Optional<Catalog> optionalCat = this.catalogRepo.findByUuid(id);
		if (optionalCat.isPresent()) {
			Catalog catalogToDelete = optionalCat.get();
			
			// Publish catalog delete notification before deletion
			catalogNotificationService.publishCatalogDeleteNotification(catalogToDelete);
			
			this.catalogRepo.delete(catalogToDelete);
		}
		return null;
	}
	
    public String findByUuidEager(String id) {
      
      Catalog sc = this.findById(id);

      ObjectMapper mapper = new ObjectMapper();
      // Registering Hibernate4Module to support lazy objects
      // this will fetch all lazy objects before marshaling
      mapper.registerModule(new Hibernate5JakartaModule());
      String res;
      try {
        res = mapper.writeValueAsString( sc );
      } catch (JsonProcessingException e) {
        // TODO Auto-generated catch block
        e.printStackTrace();
        return "{}";
      }
      
      return res;
    }  
    
    public String findAllEager() {
      
      List<Catalog> oids = (List<Catalog>) this.catalogRepo.findByOrderByName();
      ObjectMapper mapper = new ObjectMapper();
      // Registering Hibernate4Module to support lazy objects
      // this will fetch all lazy objects before marshaling
      mapper.registerModule(new Hibernate5JakartaModule());
      String res;
      try {
        res = mapper.writeValueAsString( oids );
      } catch (JsonProcessingException e) {
        // TODO Auto-generated catch block
        e.printStackTrace();
        return "{}";
      }
      
      return res;
    }
    
    public String findByNameEager(String aname) {
      Catalog sc = this.findByName(aname);

      ObjectMapper mapper = new ObjectMapper();
      // Registering Hibernate4Module to support lazy objects
      // this will fetch all lazy objects before marshaling
      mapper.registerModule(new Hibernate5JakartaModule());
      String res;
      try {
        res = mapper.writeValueAsString( sc );
      } catch (JsonProcessingException e) {
        // TODO Auto-generated catch block
        e.printStackTrace();
        return "{}";
      }
      
      return res;
           
    }

	public Catalog updateCatalog(String id, CatalogUpdate Catalog) {

		Optional<Catalog> optSC = catalogRepo.findByUuid(id);
		if (optSC == null) {
			return null;
		}
		Catalog sc = optSC.get();
		sc = updateCatalogDataFromAPICall(sc, Catalog);
		return this.catalogRepo.save(sc);
	}

	public Catalog updateCatalogDataFromAPICall(Catalog sc, CatalogUpdate Catalog) {
		
		if (Catalog.getName()!=null){
			sc.setName(Catalog.getName());			
		}
		if (Catalog.getDescription()!=null){
			sc.setDescription(Catalog.getDescription());			
		}
		if (Catalog.getLifecycleStatus() != null) {
			sc.setLifecycleStatusEnum(ELifecycle.getEnum(Catalog.getLifecycleStatus()));
		}
		if (Catalog.getVersion() != null) {
			sc.setVersion(Catalog.getVersion());
		}
		sc.setLastUpdate(OffsetDateTime.now(ZoneOffset.UTC));
		TimePeriod tp = new TimePeriod();
		if (Catalog.getValidFor() != null) {
			tp.setStartDateTime(Catalog.getValidFor().getStartDateTime());
			tp.setEndDateTime(Catalog.getValidFor().getEndDateTime());
			sc.setValidFor(tp);
		}

		// add any new category
		if (Catalog.getCategory() != null) {

			sc.getCategoryObj().clear();
			for (CategoryRef scref : Catalog.getCategory()) {
				Category servcat = this.categRepoService.findByUuid(scref.getId());
				sc.addCategory(servcat);
			}
		}

		return sc;

	}

	public Catalog updateCatalog(Catalog scatalog) {
		return this.catalogRepo.save(scatalog);
	}
	
	   /**
     * return recursively all categories in catalog
     * @param catalogName
     */
    @Transactional
    public String findAllCategoriesByCatalogName(String catalogName) {
      String res="[]";

      Optional<Catalog> scopt = this.catalogRepo.findByName(catalogName);
      
      if (scopt.isEmpty() ) {
          return res;       
      }
      
      Catalog sc = scopt.get();
      
      sc.getCategoryRefs();


      List<Category> allcategories = this.getCategories( sc.getCategoryRefs());
      
      ObjectMapper mapper = new ObjectMapper();
      // Registering Hibernate4Module to support lazy objects
      // this will fetch all lazy objects before marshaling
      mapper.registerModule(new Hibernate5JakartaModule());     
      
      
      try {
        res = mapper.writeValueAsString( allcategories );
      } catch (JsonProcessingException e) {
        // TODO Auto-generated catch block
        e.printStackTrace();
      }
      
      
      return res;
      
    }
    
    
    @Transactional
    private List<Category> getCategories( @Valid List<CategoryRef> list) {
      List<Category> categories = new ArrayList<Category>();
      
      for (CategoryRef c : list ) {          
        Category category = this.categRepoService.findByUuid( c.getId());
       categories.add(category);
       
       if (category.getCategoryRefs()!=null && category.getCategoryRefs().size()>0) {
           List<Category> subcategories = this.getCategories( category.getCategoryRefs() );
           categories.addAll(subcategories );//add children
       }
        
      }
      
      return categories;
    }


}
