package org.opendaylight.yang.gen.v1.urn.etsi.osm.yang.nsr.rev170208.exec.ns.config.primitive.input;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Deprecated;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.math.BigInteger;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.etsi.osm.yang.nsr.rev170208.exec.ns.config.primitive.input.vnf.list.VnfPrimitive;
import org.opendaylight.yang.gen.v1.urn.etsi.osm.yang.nsr.rev170208.exec.ns.config.primitive.input.vnf.list.VnfPrimitiveKey;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.yang.types.rev130715.Uuid;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;
import org.opendaylight.yangtools.yang.common.Uint64;

/**
 * Class that builds {@link VnfListBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 * 
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     VnfListBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new VnfListBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 * 
 * <p>
 * This pattern is supported by the immutable nature of VnfListBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 * 
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 * 
 * @see VnfListBuilder
 * @see Builder
 *
 */
public class VnfListBuilder implements Builder<VnfList> {

    private Uint64 _memberVnfIndexRef;
    private Map<VnfPrimitiveKey, VnfPrimitive> _vnfPrimitive;
    private Uuid _vnfrIdRef;
    private VnfListKey key;


    Map<Class<? extends Augmentation<VnfList>>, Augmentation<VnfList>> augmentation = Collections.emptyMap();

    public VnfListBuilder() {
    }

    public VnfListBuilder(VnfList base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<VnfList>>, Augmentation<VnfList>> aug =((AugmentationHolder<VnfList>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._memberVnfIndexRef = base.getMemberVnfIndexRef();
        this._vnfPrimitive = base.getVnfPrimitive();
        this._vnfrIdRef = base.getVnfrIdRef();
    }


    public VnfListKey key() {
        return key;
    }
    
    public Uint64 getMemberVnfIndexRef() {
        return _memberVnfIndexRef;
    }
    
    public Map<VnfPrimitiveKey, VnfPrimitive> getVnfPrimitive() {
        return _vnfPrimitive;
    }
    
    public Uuid getVnfrIdRef() {
        return _vnfrIdRef;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<VnfList>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(Objects.requireNonNull(augmentationType));
    }

    public VnfListBuilder withKey(final VnfListKey key) {
        this.key = key;
        return this;
    }
    
    public VnfListBuilder setMemberVnfIndexRef(final Uint64 value) {
        this._memberVnfIndexRef = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setMemberVnfIndexRef(Uint64)} instead.
     */
    @Deprecated(forRemoval = true)
    public VnfListBuilder setMemberVnfIndexRef(final BigInteger value) {
        return setMemberVnfIndexRef(CodeHelpers.compatUint(value));
    }
    public VnfListBuilder setVnfPrimitive(final Map<VnfPrimitiveKey, VnfPrimitive> values) {
        this._vnfPrimitive = values;
        return this;
    }
    
    /**
      * Utility migration setter.
      *
      * <b>IMPORTANT NOTE</b>: This method does not completely match previous mechanics, as the list is processed as
      *                        during this method's execution. Any future modifications of the list are <b>NOT</b>
      *                        reflected in this builder nor its products.
      *
      * @param values Legacy List of values
      * @return this builder
      * @throws IllegalArgumentException if the list contains entries with the same key
      * @throws NullPointerException if the list contains a null entry
      * @deprecated Use {#link #setVnfPrimitive(Map)} instead.
      */
    @Deprecated(forRemoval = true)
    public VnfListBuilder setVnfPrimitive(final List<VnfPrimitive> values) {
        return setVnfPrimitive(CodeHelpers.compatMap(values));
    }
    
    public VnfListBuilder setVnfrIdRef(final Uuid value) {
        this._vnfrIdRef = value;
        return this;
    }
    
    /**
      * Add an augmentation to this builder's product.
      *
      * @param augmentation augmentation to be added
      * @return this builder
      * @throws NullPointerException if {@code augmentation} is null
      */
    public VnfListBuilder addAugmentation(Augmentation<VnfList> augmentation) {
        return doAddAugmentation(augmentation.implementedInterface(), augmentation);
    }
    
    /**
      * Add or remove an augmentation to this builder's product.
      *
      * @param augmentationType augmentation type to be added or removed
      * @param augmentationValue augmentation value, null if the augmentation type should be removed
      * @return this builder
      * @deprecated Use either {@link #addAugmentation(Augmentation)} or {@link #removeAugmentation(Class)} instead.
      */
    @Deprecated(forRemoval = true)
    public VnfListBuilder addAugmentation(Class<? extends Augmentation<VnfList>> augmentationType, Augmentation<VnfList> augmentationValue) {
        return augmentationValue == null ? removeAugmentation(augmentationType) : doAddAugmentation(augmentationType, augmentationValue);
    }
    
    /**
      * Remove an augmentation from this builder's product. If this builder does not track such an augmentation
      * type, this method does nothing.
      *
      * @param augmentationType augmentation type to be removed
      * @return this builder
      */
    public VnfListBuilder removeAugmentation(Class<? extends Augmentation<VnfList>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }
    
    private VnfListBuilder doAddAugmentation(Class<? extends Augmentation<VnfList>> augmentationType, Augmentation<VnfList> augmentationValue) {
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }

    @Override
    public VnfList build() {
        return new VnfListImpl(this);
    }

    private static final class VnfListImpl
        extends AbstractAugmentable<VnfList>
        implements VnfList {
    
        private final Uint64 _memberVnfIndexRef;
        private final Map<VnfPrimitiveKey, VnfPrimitive> _vnfPrimitive;
        private final Uuid _vnfrIdRef;
        private final VnfListKey key;
    
        VnfListImpl(VnfListBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new VnfListKey(base.getMemberVnfIndexRef());
            }
            this._memberVnfIndexRef = key.getMemberVnfIndexRef();
            this._vnfPrimitive = CodeHelpers.emptyToNull(base.getVnfPrimitive());
            this._vnfrIdRef = base.getVnfrIdRef();
        }
    
        @Override
        public VnfListKey key() {
            return key;
        }
        
        @Override
        public Uint64 getMemberVnfIndexRef() {
            return _memberVnfIndexRef;
        }
        
        @Override
        public Map<VnfPrimitiveKey, VnfPrimitive> getVnfPrimitive() {
            return _vnfPrimitive;
        }
        
        @Override
        public Uuid getVnfrIdRef() {
            return _vnfrIdRef;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_memberVnfIndexRef);
            result = prime * result + Objects.hashCode(_vnfPrimitive);
            result = prime * result + Objects.hashCode(_vnfrIdRef);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!VnfList.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            VnfList other = (VnfList)obj;
            if (!Objects.equals(_memberVnfIndexRef, other.getMemberVnfIndexRef())) {
                return false;
            }
            if (!Objects.equals(_vnfPrimitive, other.getVnfPrimitive())) {
                return false;
            }
            if (!Objects.equals(_vnfrIdRef, other.getVnfrIdRef())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                VnfListImpl otherImpl = (VnfListImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<VnfList>>, Augmentation<VnfList>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("VnfList");
            CodeHelpers.appendValue(helper, "_memberVnfIndexRef", _memberVnfIndexRef);
            CodeHelpers.appendValue(helper, "_vnfPrimitive", _vnfPrimitive);
            CodeHelpers.appendValue(helper, "_vnfrIdRef", _vnfrIdRef);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
