package org.opendaylight.yang.gen.v1.urn.etsi.osm.yang.mano.types.rev170208.vca.configuration;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Deprecated;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.etsi.osm.yang.mano.types.rev170208.vca.configuration.config.primitive.Parameter;
import org.opendaylight.yang.gen.v1.urn.etsi.osm.yang.mano.types.rev170208.vca.configuration.config.primitive.ParameterKey;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link ConfigPrimitiveBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 * 
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     ConfigPrimitiveBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new ConfigPrimitiveBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 * 
 * <p>
 * This pattern is supported by the immutable nature of ConfigPrimitiveBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 * 
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 * 
 * @see ConfigPrimitiveBuilder
 * @see Builder
 *
 */
public class ConfigPrimitiveBuilder implements Builder<ConfigPrimitive> {

    private String _executionEnvironmentPrimitive;
    private Object _executionEnvironmentRef;
    private String _name;
    private Map<ParameterKey, Parameter> _parameter;
    private String _userDefinedScript;
    private ConfigPrimitiveKey key;


    Map<Class<? extends Augmentation<ConfigPrimitive>>, Augmentation<ConfigPrimitive>> augmentation = Collections.emptyMap();

    public ConfigPrimitiveBuilder() {
    }

    public ConfigPrimitiveBuilder(ConfigPrimitive base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<ConfigPrimitive>>, Augmentation<ConfigPrimitive>> aug =((AugmentationHolder<ConfigPrimitive>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._name = base.getName();
        this._executionEnvironmentPrimitive = base.getExecutionEnvironmentPrimitive();
        this._executionEnvironmentRef = base.getExecutionEnvironmentRef();
        this._parameter = base.getParameter();
        this._userDefinedScript = base.getUserDefinedScript();
    }


    public ConfigPrimitiveKey key() {
        return key;
    }
    
    public String getExecutionEnvironmentPrimitive() {
        return _executionEnvironmentPrimitive;
    }
    
    public Object getExecutionEnvironmentRef() {
        return _executionEnvironmentRef;
    }
    
    public String getName() {
        return _name;
    }
    
    public Map<ParameterKey, Parameter> getParameter() {
        return _parameter;
    }
    
    public String getUserDefinedScript() {
        return _userDefinedScript;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<ConfigPrimitive>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(Objects.requireNonNull(augmentationType));
    }

    public ConfigPrimitiveBuilder withKey(final ConfigPrimitiveKey key) {
        this.key = key;
        return this;
    }
    
    public ConfigPrimitiveBuilder setExecutionEnvironmentPrimitive(final String value) {
        this._executionEnvironmentPrimitive = value;
        return this;
    }
    
    public ConfigPrimitiveBuilder setExecutionEnvironmentRef(final Object value) {
        this._executionEnvironmentRef = value;
        return this;
    }
    
    public ConfigPrimitiveBuilder setName(final String value) {
        this._name = value;
        return this;
    }
    public ConfigPrimitiveBuilder setParameter(final Map<ParameterKey, Parameter> values) {
        this._parameter = values;
        return this;
    }
    
    /**
      * Utility migration setter.
      *
      * <b>IMPORTANT NOTE</b>: This method does not completely match previous mechanics, as the list is processed as
      *                        during this method's execution. Any future modifications of the list are <b>NOT</b>
      *                        reflected in this builder nor its products.
      *
      * @param values Legacy List of values
      * @return this builder
      * @throws IllegalArgumentException if the list contains entries with the same key
      * @throws NullPointerException if the list contains a null entry
      * @deprecated Use {#link #setParameter(Map)} instead.
      */
    @Deprecated(forRemoval = true)
    public ConfigPrimitiveBuilder setParameter(final List<Parameter> values) {
        return setParameter(CodeHelpers.compatMap(values));
    }
    
    public ConfigPrimitiveBuilder setUserDefinedScript(final String value) {
        this._userDefinedScript = value;
        return this;
    }
    
    /**
      * Add an augmentation to this builder's product.
      *
      * @param augmentation augmentation to be added
      * @return this builder
      * @throws NullPointerException if {@code augmentation} is null
      */
    public ConfigPrimitiveBuilder addAugmentation(Augmentation<ConfigPrimitive> augmentation) {
        return doAddAugmentation(augmentation.implementedInterface(), augmentation);
    }
    
    /**
      * Add or remove an augmentation to this builder's product.
      *
      * @param augmentationType augmentation type to be added or removed
      * @param augmentationValue augmentation value, null if the augmentation type should be removed
      * @return this builder
      * @deprecated Use either {@link #addAugmentation(Augmentation)} or {@link #removeAugmentation(Class)} instead.
      */
    @Deprecated(forRemoval = true)
    public ConfigPrimitiveBuilder addAugmentation(Class<? extends Augmentation<ConfigPrimitive>> augmentationType, Augmentation<ConfigPrimitive> augmentationValue) {
        return augmentationValue == null ? removeAugmentation(augmentationType) : doAddAugmentation(augmentationType, augmentationValue);
    }
    
    /**
      * Remove an augmentation from this builder's product. If this builder does not track such an augmentation
      * type, this method does nothing.
      *
      * @param augmentationType augmentation type to be removed
      * @return this builder
      */
    public ConfigPrimitiveBuilder removeAugmentation(Class<? extends Augmentation<ConfigPrimitive>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }
    
    private ConfigPrimitiveBuilder doAddAugmentation(Class<? extends Augmentation<ConfigPrimitive>> augmentationType, Augmentation<ConfigPrimitive> augmentationValue) {
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }

    @Override
    public ConfigPrimitive build() {
        return new ConfigPrimitiveImpl(this);
    }

    private static final class ConfigPrimitiveImpl
        extends AbstractAugmentable<ConfigPrimitive>
        implements ConfigPrimitive {
    
        private final String _executionEnvironmentPrimitive;
        private final Object _executionEnvironmentRef;
        private final String _name;
        private final Map<ParameterKey, Parameter> _parameter;
        private final String _userDefinedScript;
        private final ConfigPrimitiveKey key;
    
        ConfigPrimitiveImpl(ConfigPrimitiveBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new ConfigPrimitiveKey(base.getName());
            }
            this._name = key.getName();
            this._executionEnvironmentPrimitive = base.getExecutionEnvironmentPrimitive();
            this._executionEnvironmentRef = base.getExecutionEnvironmentRef();
            this._parameter = CodeHelpers.emptyToNull(base.getParameter());
            this._userDefinedScript = base.getUserDefinedScript();
        }
    
        @Override
        public ConfigPrimitiveKey key() {
            return key;
        }
        
        @Override
        public String getExecutionEnvironmentPrimitive() {
            return _executionEnvironmentPrimitive;
        }
        
        @Override
        public Object getExecutionEnvironmentRef() {
            return _executionEnvironmentRef;
        }
        
        @Override
        public String getName() {
            return _name;
        }
        
        @Override
        public Map<ParameterKey, Parameter> getParameter() {
            return _parameter;
        }
        
        @Override
        public String getUserDefinedScript() {
            return _userDefinedScript;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_executionEnvironmentPrimitive);
            result = prime * result + Objects.hashCode(_executionEnvironmentRef);
            result = prime * result + Objects.hashCode(_name);
            result = prime * result + Objects.hashCode(_parameter);
            result = prime * result + Objects.hashCode(_userDefinedScript);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!ConfigPrimitive.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            ConfigPrimitive other = (ConfigPrimitive)obj;
            if (!Objects.equals(_executionEnvironmentPrimitive, other.getExecutionEnvironmentPrimitive())) {
                return false;
            }
            if (!Objects.equals(_executionEnvironmentRef, other.getExecutionEnvironmentRef())) {
                return false;
            }
            if (!Objects.equals(_name, other.getName())) {
                return false;
            }
            if (!Objects.equals(_parameter, other.getParameter())) {
                return false;
            }
            if (!Objects.equals(_userDefinedScript, other.getUserDefinedScript())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                ConfigPrimitiveImpl otherImpl = (ConfigPrimitiveImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<ConfigPrimitive>>, Augmentation<ConfigPrimitive>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("ConfigPrimitive");
            CodeHelpers.appendValue(helper, "_executionEnvironmentPrimitive", _executionEnvironmentPrimitive);
            CodeHelpers.appendValue(helper, "_executionEnvironmentRef", _executionEnvironmentRef);
            CodeHelpers.appendValue(helper, "_name", _name);
            CodeHelpers.appendValue(helper, "_parameter", _parameter);
            CodeHelpers.appendValue(helper, "_userDefinedScript", _userDefinedScript);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
