package org.opendaylight.yang.gen.v1.urn.etsi.nfv.yang.etsi.nfv.descriptors.rev190425.security.group.rule;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Deprecated;
import java.lang.Integer;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;
import org.opendaylight.yangtools.yang.common.Uint16;

/**
 * Class that builds {@link SecurityGroupRuleBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 * 
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     SecurityGroupRuleBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new SecurityGroupRuleBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 * 
 * <p>
 * This pattern is supported by the immutable nature of SecurityGroupRuleBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 * 
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 * 
 * @see SecurityGroupRuleBuilder
 * @see Builder
 *
 */
public class SecurityGroupRuleBuilder implements Builder<SecurityGroupRule> {

    private String _description;
    private SecurityGroupRule.Direction _direction;
    private SecurityGroupRule.EtherType _etherType;
    private String _id;
    private Uint16 _portRangeMax;
    private Uint16 _portRangeMin;
    private SecurityGroupRule.Protocol _protocol;
    private SecurityGroupRuleKey key;


    Map<Class<? extends Augmentation<SecurityGroupRule>>, Augmentation<SecurityGroupRule>> augmentation = Collections.emptyMap();

    public SecurityGroupRuleBuilder() {
    }

    public SecurityGroupRuleBuilder(SecurityGroupRule base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<SecurityGroupRule>>, Augmentation<SecurityGroupRule>> aug =((AugmentationHolder<SecurityGroupRule>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._id = base.getId();
        this._description = base.getDescription();
        this._direction = base.getDirection();
        this._etherType = base.getEtherType();
        this._portRangeMax = base.getPortRangeMax();
        this._portRangeMin = base.getPortRangeMin();
        this._protocol = base.getProtocol();
    }


    public SecurityGroupRuleKey key() {
        return key;
    }
    
    public String getDescription() {
        return _description;
    }
    
    public SecurityGroupRule.Direction getDirection() {
        return _direction;
    }
    
    public SecurityGroupRule.EtherType getEtherType() {
        return _etherType;
    }
    
    public String getId() {
        return _id;
    }
    
    public Uint16 getPortRangeMax() {
        return _portRangeMax;
    }
    
    public Uint16 getPortRangeMin() {
        return _portRangeMin;
    }
    
    public SecurityGroupRule.Protocol getProtocol() {
        return _protocol;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<SecurityGroupRule>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(Objects.requireNonNull(augmentationType));
    }

    public SecurityGroupRuleBuilder withKey(final SecurityGroupRuleKey key) {
        this.key = key;
        return this;
    }
    
    public SecurityGroupRuleBuilder setDescription(final String value) {
        this._description = value;
        return this;
    }
    
    public SecurityGroupRuleBuilder setDirection(final SecurityGroupRule.Direction value) {
        this._direction = value;
        return this;
    }
    
    public SecurityGroupRuleBuilder setEtherType(final SecurityGroupRule.EtherType value) {
        this._etherType = value;
        return this;
    }
    
    public SecurityGroupRuleBuilder setId(final String value) {
        this._id = value;
        return this;
    }
    
    public SecurityGroupRuleBuilder setPortRangeMax(final Uint16 value) {
        this._portRangeMax = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setPortRangeMax(Uint16)} instead.
     */
    @Deprecated(forRemoval = true)
    public SecurityGroupRuleBuilder setPortRangeMax(final Integer value) {
        return setPortRangeMax(CodeHelpers.compatUint(value));
    }
    
    public SecurityGroupRuleBuilder setPortRangeMin(final Uint16 value) {
        this._portRangeMin = value;
        return this;
    }
    
    /**
     * Utility migration setter.
     *
     * @param value field value in legacy type
     * @return this builder
     * @deprecated Use {#link setPortRangeMin(Uint16)} instead.
     */
    @Deprecated(forRemoval = true)
    public SecurityGroupRuleBuilder setPortRangeMin(final Integer value) {
        return setPortRangeMin(CodeHelpers.compatUint(value));
    }
    
    public SecurityGroupRuleBuilder setProtocol(final SecurityGroupRule.Protocol value) {
        this._protocol = value;
        return this;
    }
    
    /**
      * Add an augmentation to this builder's product.
      *
      * @param augmentation augmentation to be added
      * @return this builder
      * @throws NullPointerException if {@code augmentation} is null
      */
    public SecurityGroupRuleBuilder addAugmentation(Augmentation<SecurityGroupRule> augmentation) {
        return doAddAugmentation(augmentation.implementedInterface(), augmentation);
    }
    
    /**
      * Add or remove an augmentation to this builder's product.
      *
      * @param augmentationType augmentation type to be added or removed
      * @param augmentationValue augmentation value, null if the augmentation type should be removed
      * @return this builder
      * @deprecated Use either {@link #addAugmentation(Augmentation)} or {@link #removeAugmentation(Class)} instead.
      */
    @Deprecated(forRemoval = true)
    public SecurityGroupRuleBuilder addAugmentation(Class<? extends Augmentation<SecurityGroupRule>> augmentationType, Augmentation<SecurityGroupRule> augmentationValue) {
        return augmentationValue == null ? removeAugmentation(augmentationType) : doAddAugmentation(augmentationType, augmentationValue);
    }
    
    /**
      * Remove an augmentation from this builder's product. If this builder does not track such an augmentation
      * type, this method does nothing.
      *
      * @param augmentationType augmentation type to be removed
      * @return this builder
      */
    public SecurityGroupRuleBuilder removeAugmentation(Class<? extends Augmentation<SecurityGroupRule>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }
    
    private SecurityGroupRuleBuilder doAddAugmentation(Class<? extends Augmentation<SecurityGroupRule>> augmentationType, Augmentation<SecurityGroupRule> augmentationValue) {
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }

    @Override
    public SecurityGroupRule build() {
        return new SecurityGroupRuleImpl(this);
    }

    private static final class SecurityGroupRuleImpl
        extends AbstractAugmentable<SecurityGroupRule>
        implements SecurityGroupRule {
    
        private final String _description;
        private final SecurityGroupRule.Direction _direction;
        private final SecurityGroupRule.EtherType _etherType;
        private final String _id;
        private final Uint16 _portRangeMax;
        private final Uint16 _portRangeMin;
        private final SecurityGroupRule.Protocol _protocol;
        private final SecurityGroupRuleKey key;
    
        SecurityGroupRuleImpl(SecurityGroupRuleBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new SecurityGroupRuleKey(base.getId());
            }
            this._id = key.getId();
            this._description = base.getDescription();
            this._direction = base.getDirection();
            this._etherType = base.getEtherType();
            this._portRangeMax = base.getPortRangeMax();
            this._portRangeMin = base.getPortRangeMin();
            this._protocol = base.getProtocol();
        }
    
        @Override
        public SecurityGroupRuleKey key() {
            return key;
        }
        
        @Override
        public String getDescription() {
            return _description;
        }
        
        @Override
        public SecurityGroupRule.Direction getDirection() {
            return _direction;
        }
        
        @Override
        public SecurityGroupRule.EtherType getEtherType() {
            return _etherType;
        }
        
        @Override
        public String getId() {
            return _id;
        }
        
        @Override
        public Uint16 getPortRangeMax() {
            return _portRangeMax;
        }
        
        @Override
        public Uint16 getPortRangeMin() {
            return _portRangeMin;
        }
        
        @Override
        public SecurityGroupRule.Protocol getProtocol() {
            return _protocol;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_description);
            result = prime * result + Objects.hashCode(_direction);
            result = prime * result + Objects.hashCode(_etherType);
            result = prime * result + Objects.hashCode(_id);
            result = prime * result + Objects.hashCode(_portRangeMax);
            result = prime * result + Objects.hashCode(_portRangeMin);
            result = prime * result + Objects.hashCode(_protocol);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!SecurityGroupRule.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            SecurityGroupRule other = (SecurityGroupRule)obj;
            if (!Objects.equals(_description, other.getDescription())) {
                return false;
            }
            if (!Objects.equals(_direction, other.getDirection())) {
                return false;
            }
            if (!Objects.equals(_etherType, other.getEtherType())) {
                return false;
            }
            if (!Objects.equals(_id, other.getId())) {
                return false;
            }
            if (!Objects.equals(_portRangeMax, other.getPortRangeMax())) {
                return false;
            }
            if (!Objects.equals(_portRangeMin, other.getPortRangeMin())) {
                return false;
            }
            if (!Objects.equals(_protocol, other.getProtocol())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                SecurityGroupRuleImpl otherImpl = (SecurityGroupRuleImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<SecurityGroupRule>>, Augmentation<SecurityGroupRule>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("SecurityGroupRule");
            CodeHelpers.appendValue(helper, "_description", _description);
            CodeHelpers.appendValue(helper, "_direction", _direction);
            CodeHelpers.appendValue(helper, "_etherType", _etherType);
            CodeHelpers.appendValue(helper, "_id", _id);
            CodeHelpers.appendValue(helper, "_portRangeMax", _portRangeMax);
            CodeHelpers.appendValue(helper, "_portRangeMin", _portRangeMin);
            CodeHelpers.appendValue(helper, "_protocol", _protocol);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
